// ===================================
// BAIT ALAMER PRICING CALCULATOR - CUSTOMER PORTAL
// JavaScript Logic & Calculations
// Dynamic Pricing with Role-Based Access
// ===================================

// ==================
// PRICE LOADING
// ==================

// BASE_PRICE_MAP, SELLING_PRICE_MAP, USER_ROLE, and IS_PRICING_MANAGER are injected by PHP
// This function updates item prices from the maps
function loadDynamicPrices() {
    ITEMS_DATA.forEach(item => {
        // Load base price (company's internal price)
        if (BASE_PRICE_MAP[item.id]) {
            item.basePrice = BASE_PRICE_MAP[item.id];
        }
        
        // Load selling price (customer's price)
        if (SELLING_PRICE_MAP[item.id]) {
            item.price = SELLING_PRICE_MAP[item.id];
        } else {
            // If no selling price, default to base price
            item.price = item.basePrice || item.price;
        }
    });
    
    console.log('✅ Prices loaded - Base:', Object.keys(BASE_PRICE_MAP).length, 'Selling:', Object.keys(SELLING_PRICE_MAP).length);
}

// ==================
// DATA: All 63 Items
// ==================

const ITEMS_DATA = [
    {
        id: 1,
        category: "Gypsum Board Items",
        categoryIcon: "🏗️",
        name: "Gypsum board flat KSA",
        unit: "m²",
        price: 38000,
        inputs: ["length", "width"],
        formula: (inputs) => Math.ceil(inputs.length) * Math.ceil(inputs.width),
        requiredFields: "Length, Width"
    },
    {
        id: 2,
        category: "Gypsum Board Items",
        categoryIcon: "🏗️",
        name: "Gypsum board flat KSA L-Shape",
        unit: "m²",
        price: 38000,
        inputs: ["length", "width"],
        formula: (inputs) => ((inputs.length * 0.45 + inputs.length) * (inputs.width * 0.45 + inputs.width)),
        requiredFields: "Longest Length, Longest Width"
    },
    {
        id: 3,
        category: "Gypsum Board Items",
        categoryIcon: "🏗️",
        name: "Gypsum board flat Irani",
        unit: "m²",
        price: 36000,
        inputs: ["length", "width"],
        formula: (inputs) => Math.ceil(inputs.length) * Math.ceil(inputs.width),
        requiredFields: "Length, Width"
    },
    {
        id: 4,
        category: "Gypsum Board Items",
        categoryIcon: "🏗️",
        name: "Gypsum board flat Irani L-Shape",
        unit: "m²",
        price: 36000,
        inputs: ["length", "width"],
        formula: (inputs) => ((inputs.length * 0.45 + inputs.length) * (inputs.width * 0.45 + inputs.width)),
        requiredFields: "Longest Length, Longest Width"
    },
    {
        id: 5,
        category: "Gypsum Board Items",
        categoryIcon: "🏗️",
        name: "Gypsum board box KSA",
        unit: "m²",
        price: 20000,
        inputs: ["length", "width"],
        formula: (inputs) => {
            const area = inputs.length * inputs.width;
            if (area < 30) return area + 2;
            if (area <= 60) return area + 4;
            return area + 6;
        },
        requiredFields: "Length, Width"
    },
    {
        id: 6,
        category: "Gypsum Board Items",
        categoryIcon: "🏗️",
        name: "Gypsum board box KSA L-Shape",
        unit: "m²",
        price: 20000,
        inputs: ["length", "width"],
        formula: (inputs) => {
            const area = (inputs.length * 0.45 + inputs.length) * (inputs.width * 0.45 + inputs.width);
            if (area < 30) return area + 2;
            if (area <= 60) return area + 4;
            return area + 6;
        },
        requiredFields: "Longest Length, Longest Width"
    },
    {
        id: 7,
        category: "Gypsum Board Items",
        categoryIcon: "🏗️",
        name: "Gypsum board box Irani",
        unit: "m²",
        price: 18000,
        inputs: ["length", "width"],
        formula: (inputs) => {
            const area = inputs.length * inputs.width;
            if (area < 30) return area + 2;
            if (area <= 60) return area + 4;
            return area + 6;
        },
        requiredFields: "Length, Width"
    },
    {
        id: 8,
        category: "Gypsum Board Items",
        categoryIcon: "🏗️",
        name: "Gypsum board box Irani L-Shape",
        unit: "m²",
        price: 18000,
        inputs: ["length", "width"],
        formula: (inputs) => {
            const area = (inputs.length * 0.45 + inputs.length) * (inputs.width * 0.45 + inputs.width);
            if (area < 30) return area + 2;
            if (area <= 60) return area + 4;
            return area + 6;
        },
        requiredFields: "Longest Length, Longest Width"
    },
    {
        id: 9,
        category: "Gypsum Board Items",
        categoryIcon: "🏗️",
        name: "Gypsum board curtains",
        unit: "m",
        price: 8500,
        inputs: ["length"],
        formula: (inputs) => (inputs.length * 2) + 2,
        requiredFields: "Length"
    },
    {
        id: 10,
        category: "Gypsum Decorative & Lighting",
        categoryIcon: "✨",
        name: "Hidden light",
        unit: "m",
        price: 4500,
        inputs: ["length"],
        formula: (inputs) => inputs.length,
        requiredFields: "Length"
    },
    {
        id: 11,
        category: "Spot Lights",
        categoryIcon: "💡",
        name: "Hidden spot light single",
        unit: "piece",
        price: 15000,
        inputs: ["quantity"],
        formula: (inputs) => inputs.quantity,
        requiredFields: "Quantity"
    },
    {
        id: 12,
        category: "Spot Lights",
        categoryIcon: "💡",
        name: "Hidden spot light dual",
        unit: "piece",
        price: 28000,
        inputs: ["quantity"],
        formula: (inputs) => inputs.quantity,
        requiredFields: "Quantity"
    },
    {
        id: 13,
        category: "Spot Lights",
        categoryIcon: "💡",
        name: "Hidden spot light black",
        unit: "piece",
        price: 18000,
        inputs: ["quantity"],
        formula: (inputs) => inputs.quantity,
        requiredFields: "Quantity"
    },
    {
        id: 14,
        category: "Profile LED (Standard)",
        categoryIcon: "🔦",
        name: "profile led 2 cm",
        unit: "m",
        price: 28000,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length / 3) * 3,
        requiredFields: "Length"
    },
    {
        id: 15,
        category: "Profile LED (Standard)",
        categoryIcon: "🔦",
        name: "profile led 3cm",
        unit: "m",
        price: 30000,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length / 3) * 3,
        requiredFields: "Length"
    },
    {
        id: 16,
        category: "Profile LED (Standard)",
        categoryIcon: "🔦",
        name: "profile led 5 cm",
        unit: "m",
        price: 33000,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length / 3) * 3,
        requiredFields: "Length"
    },
    {
        id: 17,
        category: "Profile LED (Black)",
        categoryIcon: "🔦",
        name: "profile led 2 cm black",
        unit: "m",
        price: 30000,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length / 3) * 3,
        requiredFields: "Length"
    },
    {
        id: 18,
        category: "Profile LED (Black)",
        categoryIcon: "🔦",
        name: "profile led 3 cm black",
        unit: "m",
        price: 33000,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length / 3) * 3,
        requiredFields: "Length"
    },
    {
        id: 19,
        category: "Profile LED (Black)",
        categoryIcon: "🔦",
        name: "profile led 5 cm black",
        unit: "m",
        price: 36000,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length / 3) * 3,
        requiredFields: "Length"
    },
    {
        id: 20,
        category: "Profile Magnetic",
        categoryIcon: "🧲",
        name: "profile magnetic 2 cm",
        unit: "m",
        price: 32000,
        inputs: ["length"],
        formula: (inputs) => inputs.length,
        requiredFields: "Length"
    },
    {
        id: 21,
        category: "Profile Magnetic",
        categoryIcon: "🧲",
        name: "profile magnetic 3 cm",
        unit: "m",
        price: 36000,
        inputs: ["length"],
        formula: (inputs) => inputs.length,
        requiredFields: "Length"
    },
    {
        id: 22,
        category: "Profile Magnetic",
        categoryIcon: "🧲",
        name: "profile magnetic 7 cm",
        unit: "m",
        price: 40000,
        inputs: ["length"],
        formula: (inputs) => inputs.length,
        requiredFields: "Length"
    },
    {
        id: 23,
        category: "Power Supply",
        categoryIcon: "🔌",
        name: "profile led power supply",
        unit: "piece",
        price: 27000,
        inputs: ["quantity"],
        formula: (inputs) => inputs.quantity,
        requiredFields: "Quantity"
    },
    {
        id: 24,
        category: "Power Supply",
        categoryIcon: "🔌",
        name: "profile magnetic power supply",
        unit: "piece",
        price: 27000,
        inputs: ["quantity"],
        formula: (inputs) => inputs.quantity,
        requiredFields: "Quantity"
    },
    {
        id: 25,
        category: "Magnetic LED",
        categoryIcon: "💫",
        name: "magnetic fixed led",
        unit: "piece",
        price: 35000,
        inputs: ["quantity"],
        formula: (inputs) => inputs.quantity,
        requiredFields: "Quantity"
    },
    {
        id: 26,
        category: "Magnetic LED",
        categoryIcon: "💫",
        name: "magnetic movable led",
        unit: "piece",
        price: 37000,
        inputs: ["quantity"],
        formula: (inputs) => inputs.quantity,
        requiredFields: "Quantity"
    },
    {
        id: 27,
        category: "Wall Cladding",
        categoryIcon: "🎨",
        name: "wood plastic composite",
        unit: "m²",
        price: 27000,
        inputs: ["length", "width"],
        formula: (inputs) => Math.ceil(inputs.length) * Math.ceil(inputs.width),
        requiredFields: "Length, Width"
    },
    {
        id: 28,
        category: "Wall Cladding",
        categoryIcon: "🎨",
        name: "wallpaper",
        unit: "m²",
        price: 35000,
        inputs: ["length", "width"],
        formula: (inputs) => Math.ceil(inputs.length) * Math.ceil(inputs.width),
        requiredFields: "Length, Width"
    },
    {
        id: 29,
        category: "Wall Cladding",
        categoryIcon: "🎨",
        name: "skinrock",
        unit: "m²",
        price: 79000,
        inputs: ["length", "width"],
        formula: (inputs) => Math.ceil(inputs.length) * Math.ceil(inputs.width),
        requiredFields: "Length, Width"
    },
    {
        id: 30,
        category: "Wall Cladding",
        categoryIcon: "🎨",
        name: "rock plastic composite",
        unit: "m²",
        price: 43000,
        inputs: ["length", "width"],
        formula: (inputs) => Math.ceil(inputs.length) * Math.ceil(inputs.width),
        requiredFields: "Length, Width"
    },
    {
        id: 31,
        category: "Wall Cladding",
        categoryIcon: "🎨",
        name: "cheboard wood",
        unit: "m²",
        price: 35000,
        inputs: ["length", "width"],
        formula: (inputs) => Math.ceil(inputs.length) * Math.ceil(inputs.width),
        requiredFields: "Length, Width"
    },
    {
        id: 32,
        category: "Paint & Finishing",
        categoryIcon: "🖌️",
        name: "paint jotun",
        unit: "m²",
        price: 4500,
        inputs: ["width", "length"],
        formula: (inputs) => (inputs.width * 2 * 3) + (inputs.length * 2 * 3),
        requiredFields: "Width, Length"
    },
    {
        id: 33,
        category: "Paint & Finishing",
        categoryIcon: "🖌️",
        name: "paint with fixing wall notes",
        unit: "m²",
        price: 7500,
        inputs: ["width", "length"],
        formula: (inputs) => (inputs.width * 2 * 3) + (inputs.length * 2 * 3),
        requiredFields: "Width, Length"
    },
    {
        id: 34,
        category: "Paint & Finishing",
        categoryIcon: "🖌️",
        name: "fixing all wall notes and paint",
        unit: "m²",
        price: 9500,
        inputs: ["width", "length"],
        formula: (inputs) => (inputs.width * 2 * 3) + (inputs.length * 2 * 3),
        requiredFields: "Width, Length"
    },
    {
        id: 35,
        category: "Electrical",
        categoryIcon: "⚡",
        name: "electricity work",
        unit: "room",
        price: 50000,
        inputs: ["rooms", "avgRoomArea"],
        formula: (inputs) => inputs.avgRoomArea < 35 ? inputs.rooms : 0,
        requiredFields: "Number of rooms, Avg Room Area"
    },
    {
        id: 36,
        category: "Gypsum Decorative Panels (Flat)",
        categoryIcon: "📏",
        name: "gypsum decorative panel 3 cm flat",
        unit: "m",
        price: 3500,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length),
        requiredFields: "Length"
    },
    {
        id: 37,
        category: "Gypsum Decorative Panels (Flat)",
        categoryIcon: "📏",
        name: "gypsum decorative panel 5 cm flat",
        unit: "m",
        price: 4500,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length),
        requiredFields: "Length"
    },
    {
        id: 38,
        category: "Gypsum Decorative Panels (Flat)",
        categoryIcon: "📏",
        name: "gypsum decorative panel 7 cm flat",
        unit: "m",
        price: 6500,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length),
        requiredFields: "Length"
    },
    {
        id: 39,
        category: "Gypsum Decorative Panels (Flat)",
        categoryIcon: "📏",
        name: "gypsum decorative panel 10 cm flat",
        unit: "m",
        price: 9500,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length),
        requiredFields: "Length"
    },
    {
        id: 40,
        category: "Gypsum Decorative Panels (Flat)",
        categoryIcon: "📏",
        name: "gypsum decorative panel 12 cm flat",
        unit: "m",
        price: 11500,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length),
        requiredFields: "Length"
    },
    {
        id: 41,
        category: "Gypsum Decorative Panels (Decor)",
        categoryIcon: "✨",
        name: "gypsum decorative panel 3 cm decor",
        unit: "m",
        price: 4500,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length),
        requiredFields: "Length"
    },
    {
        id: 42,
        category: "Gypsum Decorative Panels (Decor)",
        categoryIcon: "✨",
        name: "gypsum decorative panel 5 cm decor",
        unit: "m",
        price: 6500,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length),
        requiredFields: "Length"
    },
    {
        id: 43,
        category: "Gypsum Decorative Panels (Decor)",
        categoryIcon: "✨",
        name: "gypsum decorative panel 7 cm decor",
        unit: "m",
        price: 10500,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length),
        requiredFields: "Length"
    },
    {
        id: 44,
        category: "Gypsum Decorative Panels (Decor)",
        categoryIcon: "✨",
        name: "gypsum decorative panel 10 cm decor",
        unit: "m",
        price: 12500,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length),
        requiredFields: "Length"
    },
    {
        id: 45,
        category: "Gypsum Decorative Panels (Decor)",
        categoryIcon: "✨",
        name: "gypsum decorative panel 12 cm decor",
        unit: "m",
        price: 14500,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length),
        requiredFields: "Length"
    },
    {
        id: 46,
        category: "Gypsum Decorative Panels (Decor)",
        categoryIcon: "✨",
        name: "gypsum decorative panel 5 cm ceiling",
        unit: "m",
        price: 6500,
        inputs: ["length"],
        formula: (inputs) => inputs.length * 1.5,
        requiredFields: "Length"
    },
    {
        id: 47,
        category: "Gypsum Decorative Panels (Decor)",
        categoryIcon: "✨",
        name: "gypsum decorative panel 7 cm ceiling",
        unit: "m",
        price: 10500,
        inputs: ["length"],
        formula: (inputs) => inputs.length * 1.5,
        requiredFields: "Length"
    },
    {
        id: 48,
        category: "Gypsum Decorative Panels (Decor)",
        categoryIcon: "✨",
        name: "gypsum decorative panel 10 cm ceiling",
        unit: "m",
        price: 12500,
        inputs: ["length"],
        formula: (inputs) => inputs.length * 1.5,
        requiredFields: "Length"
    },
    {
        id: 49,
        category: "Gypsum Decorative Panels (Decor)",
        categoryIcon: "✨",
        name: "gypsum decorative panel 12 cm ceiling",
        unit: "m",
        price: 14500,
        inputs: ["length"],
        formula: (inputs) => inputs.length * 1.5,
        requiredFields: "Length"
    },
    {
        id: 50,
        category: "Gypsum Decorative Panels (Decor)",
        categoryIcon: "✨",
        name: "gypsum decorative panel 14 cm angle",
        unit: "m",
        price: 21000,
        inputs: ["length"],
        formula: (inputs) => inputs.length * 1.5,
        requiredFields: "Length"
    },
    {
        id: 51,
        category: "Gypsum Decorative Panels (Decor)",
        categoryIcon: "✨",
        name: "gypsum decorative panel 18 cm angle",
        unit: "m",
        price: 24500,
        inputs: ["length"],
        formula: (inputs) => inputs.length * 1.5,
        requiredFields: "Length"
    },
    {
        id: 52,
        category: "Gypsum Decorative Panels (Decor)",
        categoryIcon: "✨",
        name: "gypsum decorative panel 21 cm angle",
        unit: "m",
        price: 29500,
        inputs: ["length"],
        formula: (inputs) => inputs.length * 1.5,
        requiredFields: "Length"
    },
    {
        id: 53,
        category: "Gypsum Decorative Panels (Decor)",
        categoryIcon: "✨",
        name: "gypsum decorative panel 24 cm angle",
        unit: "m",
        price: 33000,
        inputs: ["length"],
        formula: (inputs) => inputs.length * 1.5,
        requiredFields: "Length"
    },
    {
        id: 54,
        category: "Gypsum Decorative Panels (Decor)",
        categoryIcon: "✨",
        name: "gypsum decorative panel 20 cm royal ceiling",
        unit: "m",
        price: 50000,
        inputs: ["length"],
        formula: (inputs) => inputs.length * 2,
        requiredFields: "Length"
    },
    {
        id: 55,
        category: "Gypsum Decorative Panels (Decor)",
        categoryIcon: "✨",
        name: "gypsum decorative panel 30 cm royal ceiling",
        unit: "m",
        price: 75000,
        inputs: ["length"],
        formula: (inputs) => inputs.length * 2,
        requiredFields: "Length"
    },
    {
        id: 56,
        category: "Gypsum Decorative Panels (Decor)",
        categoryIcon: "✨",
        name: "gypsum decorative panel 10 cm royal wall",
        unit: "m",
        price: 30000,
        inputs: ["length"],
        formula: (inputs) => Math.ceil(inputs.length),
        requiredFields: "Length"
    },
    {
        id: 57,
        category: "Gypsum Decorative Panels (Royal)",
        categoryIcon: "👑",
        name: "gypsum decoration corner royal",
        unit: "piece",
        price: 20000,
        inputs: ["quantity"],
        formula: (inputs) => inputs.quantity,
        requiredFields: "Quantity"
    },
    {
        id: 58,
        category: "Gypsum Decorative Panels (Royal)",
        categoryIcon: "👑",
        name: "gypsum decoration circle ceiling royal",
        unit: "piece",
        price: 30000,
        inputs: ["quantity"],
        formula: (inputs) => inputs.quantity,
        requiredFields: "Quantity"
    },
    {
        id: 59,
        category: "Miscellaneous",
        categoryIcon: "🔩",
        name: "rawl-bolt",
        unit: "piece",
        price: 17000,
        inputs: ["quantity"],
        formula: (inputs) => inputs.quantity,
        requiredFields: "Quantity"
    },
    {
        id: 60,
        category: "Gypsum Board Decor",
        categoryIcon: "🎨",
        name: "gypsum boaard decor flat",
        unit: "m²",
        price: 38000,
        inputs: ["length", "width"],
        formula: (inputs) => (inputs.length * inputs.width) * 1.5,
        requiredFields: "Length, Width"
    },
    {
        id: 61,
        category: "Gypsum Board Decor",
        categoryIcon: "🎨",
        name: "gypsum board decor flat L-shape",
        unit: "m²",
        price: 38000,
        inputs: ["length", "width"],
        formula: (inputs) => ((inputs.length * 0.45 + inputs.length) * (inputs.width * 0.45 + inputs.width)) * 1.5,
        requiredFields: "Longest Length, Longest Width"
    },
    {
        id: 62,
        category: "Gypsum Board Decor",
        categoryIcon: "🎨",
        name: "gypsum board decor box",
        unit: "m²",
        price: 20000,
        inputs: ["length", "width"],
        formula: (inputs) => {
            const area = (inputs.length * inputs.width) * 1.5;
            if (area < 30) return area + 2;
            if (area <= 60) return area + 4;
            return area + 6;
        },
        requiredFields: "Length, Width"
    },
    {
        id: 63,
        category: "Gypsum Board Decor",
        categoryIcon: "🎨",
        name: "gypsum board decor box L shape",
        unit: "m²",
        price: 20000,
        inputs: ["length", "width"],
        formula: (inputs) => {
            const area = ((inputs.length * 0.45 + inputs.length) * (inputs.width * 0.45 + inputs.width)) * 1.5;
            if (area < 30) return area + 2;
            if (area <= 60) return area + 4;
            return area + 6;
        },
        requiredFields: "Longest Length, Longest Width"
    }
];

// ==================
// STATE MANAGEMENT
// ==================

const state = {
    selectedItems: new Set(),
    itemInputs: {},
    calculations: {},
    grandTotal: 0,
    baseGrandTotal: 0
};

// ==================
// UTILITY FUNCTIONS
// ==================

function formatCurrency(amount) {
    return Math.round(amount).toLocaleString('en-US') + ' IQD';
}

function getInputLabel(inputType) {
    const labels = {
        length: 'Length (m)',
        width: 'Width (m)',
        quantity: 'Quantity',
        rooms: 'Number of Rooms',
        avgArea: 'confirm number of rooms (only)'
    };
    return labels[inputType] || inputType;
}

// ==================
// RENDER FUNCTIONS
// ==================

function renderCategories() {
    const container = document.getElementById('categoriesContainer');
    const categories = [...new Set(ITEMS_DATA.map(item => item.category))];
    
    container.innerHTML = categories.map((category, index) => {
        const categoryItems = ITEMS_DATA.filter(item => item.category === category);
        const categoryIcon = categoryItems[0].categoryIcon;
        
        return `
            <div class="category" id="category-${index}">
                <div class="category-header" onclick="toggleCategory(${index})">
                    <div class="category-title">
                        <span class="category-icon">${categoryIcon}</span>
                        <span data-category-en="${category}">${category}</span>
                    </div>
                    <div class="category-info">
                        ${categoryItems.length} item${categoryItems.length > 1 ? 's' : ''}
                    </div>
                </div>
                <div class="category-content" id="category-content-${index}">
                    ${categoryItems.map(item => renderItem(item)).join('')}
                </div>
            </div>
        `;
    }).join('');
}

function renderItem(item) {
    // Role-based price display
    let priceDisplay;
    if (IS_PRICING_MANAGER) {
        // Pricing Manager sees base price (fixed) + editable selling price
        priceDisplay = `
            <span style="color: #fff; font-size: 12px;">Base: ${formatCurrency(item.basePrice || item.price)}</span> | 
            <input type="number" class="price-input" data-item-id="${item.id}" value="${item.price}" min="0" step="1000">
        `;
    } else {
        // Sales Consultant sees only selling price (read-only)
        priceDisplay = `<span class="price-display">${formatCurrency(item.price)}</span>`;
    }
    
    return `
        <div class="item-row" id="item-${item.id}" data-item-id="${item.id}">
            <div class="item-header">
                <input 
                    type="checkbox" 
                    class="item-checkbox" 
                    id="checkbox-${item.id}"
                    onchange="toggleItem(${item.id})"
                >
                <div class="item-info">
                    <div class="item-name">${item.name}</div>
                    <div class="item-meta">
                        <div class="item-unit">📦 Unit: <strong>${item.unit}</strong></div>
                        <div class="item-price price-cell">💰 Price: <strong>${priceDisplay}</strong></div>
                    </div>
                </div>
            </div>
            
            <div class="item-inputs" id="inputs-${item.id}">
                <div class="input-grid">
                    ${item.inputs.map(inputType => `
                        <div class="input-field">
                            <label for="input-${item.id}-${inputType}">
                                ${getInputLabel(inputType)}
                            </label>
                            <input 
                                type="number" 
                                id="input-${item.id}-${inputType}"
                                min="0"
                                step="${inputType === 'quantity' || inputType === 'rooms' ? '1' : '0.1'}"
                                placeholder="0"
                                lang="en"
                                inputmode="${inputType === 'quantity' || inputType === 'rooms' ? 'numeric' : 'decimal'}"
                                oninput="updateItemCalculation(${item.id})"
                            >
                        </div>
                    `).join('')}
                </div>
                
                <div class="item-results">
                    <div class="result-item">
                        <div class="result-label">Adjusted Quantity</div>
                        <div class="result-value" id="adjusted-qty-${item.id}">0 ${item.unit}</div>
                    </div>
                    <div class="result-item">
                        <div class="result-label">Line Total</div>
                        <div class="result-value highlight" id="line-total-${item.id}">0 IQD</div>
                    </div>
                </div>
                
                <div class="required-fields">
                    <strong>Required:</strong> ${item.requiredFields}
                </div>
            </div>
        </div>
    `;
}

// ==================
// INTERACTION HANDLERS
// ==================

function toggleCategory(index) {
    const header = document.querySelector(`#category-${index} .category-header`);
    const content = document.getElementById(`category-content-${index}`);
    
    header.classList.toggle('expanded');
    content.classList.toggle('expanded');
}

function toggleItem(itemId) {
    const checkbox = document.getElementById(`checkbox-${itemId}`);
    const itemRow = document.getElementById(`item-${itemId}`);
    const item = ITEMS_DATA.find(i => i.id === itemId);
    
    if (checkbox.checked) {
        state.selectedItems.add(itemId);
        itemRow.classList.add('selected');
        
        // Auto-expand category
        const categoryIndex = getCategoryIndex(item.category);
        const header = document.querySelector(`#category-${categoryIndex} .category-header`);
        const content = document.getElementById(`category-content-${categoryIndex}`);
        const categoryDiv = document.getElementById(`category-${categoryIndex}`);
        
        header.classList.add('expanded');
        content.classList.add('expanded');
        categoryDiv.classList.add('has-selected');
        
        // Auto-fill from header for electricity item
        if (item.isElectricity) {
            const numRooms = document.getElementById('numRooms').value;
            const avgArea = document.getElementById('avgRoomArea').value;
            if (numRooms) document.getElementById(`input-${itemId}-rooms`).value = numRooms;
            if (avgArea) document.getElementById(`input-${itemId}-avgArea`).value = avgArea;
        }
    } else {
        state.selectedItems.delete(itemId);
        itemRow.classList.remove('selected');
        
        // Check if category still has selected items
        const categoryIndex = getCategoryIndex(item.category);
        const categoryDiv = document.getElementById(`category-${categoryIndex}`);
        const categoryItems = ITEMS_DATA.filter(i => i.category === item.category);
        const hasSelected = categoryItems.some(i => state.selectedItems.has(i.id));
        
        if (!hasSelected) {
            categoryDiv.classList.remove('has-selected');
        }
        
        // Clear inputs
        state.itemInputs[itemId] = {};
        state.calculations[itemId] = null;
    }
    
    updateConditionalRequirements();
    updateItemCalculation(itemId);
}

function getCategoryIndex(categoryName) {
    const categories = [...new Set(ITEMS_DATA.map(item => item.category))];
    return categories.indexOf(categoryName);
}

function updateItemCalculation(itemId) {
    const item = ITEMS_DATA.find(i => i.id === itemId);
    if (!state.selectedItems.has(itemId)) return;
    
    // Gather inputs
    const inputs = {};
    let hasAllInputs = true;
    
    item.inputs.forEach(inputType => {
        const inputField = document.getElementById(`input-${itemId}-${inputType}`);
        const inputValue = inputField.value.trim();
        
        // Parse the value - handle both integer and decimal inputs
        let value = 0;
        if (inputValue !== '' && inputValue !== '.') {
            value = parseFloat(inputValue);
            // Check if it's a valid number
            if (isNaN(value) || value < 0) {
                value = 0;
            }
        }
        
        inputs[inputType] = value;
        if (value === 0 || value === null || isNaN(value)) {
            hasAllInputs = false;
        }
    });
    
    state.itemInputs[itemId] = inputs;
    
    // Get current price (may have been edited by pricing manager)
    let currentPrice = item.price;
    if (IS_PRICING_MANAGER) {
        const priceInput = document.querySelector(`.price-input[data-item-id="${itemId}"]`);
        if (priceInput) {
            const inputPrice = parseFloat(priceInput.value);
            if (!isNaN(inputPrice) && inputPrice >= 0) {
                currentPrice = inputPrice;
            }
        }
    }
    
    // Calculate adjusted quantity
    const adjustedQty = hasAllInputs ? item.formula(inputs) : 0;
    const lineTotal = adjustedQty * currentPrice;
    
    state.calculations[itemId] = {
        adjustedQty,
        lineTotal
    };
    
    // Update display - show decimals if not whole number
    let displayQty;
    if (adjustedQty === 0) {
        displayQty = '0';
    } else if (adjustedQty % 1 === 0) {
        displayQty = adjustedQty.toFixed(0);
    } else {
        // Show up to 2 decimal places, remove trailing zeros
        displayQty = parseFloat(adjustedQty.toFixed(2)).toString();
    }
    
    document.getElementById(`adjusted-qty-${itemId}`).textContent = 
        `${displayQty} ${item.unit}`;
    document.getElementById(`line-total-${itemId}`).textContent = 
        formatCurrency(lineTotal);
}

function updateConditionalRequirements() {
    // Check if any paint items selected
    const paintSelected = Array.from(state.selectedItems).some(id => {
        const item = ITEMS_DATA.find(i => i.id === id);
        return item.isPaint;
    });
    
    // Check if electricity item selected
    const electricitySelected = Array.from(state.selectedItems).some(id => {
        const item = ITEMS_DATA.find(i => i.id === id);
        return item.isElectricity;
    });
    
    // Update asterisks
    const paintAsterisk = document.getElementById('paint-asterisk');
    const areaAsterisk = document.getElementById('area-asterisk');
    const roomsAsterisk = document.getElementById('rooms-asterisk');
    
    if (paintSelected) {
        paintAsterisk.classList.add('active');
        roomsAsterisk.classList.add('active');
    } else {
        paintAsterisk.classList.remove('active');
        if (!electricitySelected) {
            roomsAsterisk.classList.remove('active');
        }
    }
    
    if (electricitySelected) {
        areaAsterisk.classList.add('active');
        roomsAsterisk.classList.add('active');
    } else {
        areaAsterisk.classList.remove('active');
        if (!paintSelected) {
            roomsAsterisk.classList.remove('active');
        }
    }
}

// ==================
// VALIDATION
// ==================

function validateInputs() {
    const errors = [];
    
    // Always mandatory
    const customerName = document.getElementById('customerName').value.trim();
    const siteLocation = document.getElementById('siteLocation').value.trim();
    
    if (!customerName) errors.push('Customer Name is required');
    if (!siteLocation) errors.push('Site Location is required');
    
    // Check for paint items
    const paintSelected = Array.from(state.selectedItems).some(id => {
        const item = ITEMS_DATA.find(i => i.id === id);
        return item.isPaint;
    });
    
    // Check for electricity items
    const electricitySelected = Array.from(state.selectedItems).some(id => {
        const item = ITEMS_DATA.find(i => i.id === id);
        return item.isElectricity;
    });
    
    // Conditional validations
    if (paintSelected || electricitySelected) {
        const numRooms = document.getElementById('numRooms').value;
        if (!numRooms || numRooms == 0) {
            errors.push('Number of Rooms is required (paint or electricity items selected)');
        }
    }
    
    if (electricitySelected) {
        const avgArea = document.getElementById('avgRoomArea').value;
        if (!avgArea || avgArea == 0) {
            errors.push('Average Room Area is required (electricity item selected)');
        }
    }
    
    if (paintSelected) {
        const paintCode = document.getElementById('paintColorCode').value.trim();
        if (!paintCode) {
            errors.push('Paint Color Code is required (paint items selected)');
        }
    }
    
    // Item-level validations
    state.selectedItems.forEach(itemId => {
        const item = ITEMS_DATA.find(i => i.id === itemId);
        const inputs = state.itemInputs[itemId] || {};
        
        item.inputs.forEach(inputType => {
            if (!inputs[inputType] || inputs[inputType] === 0) {
                errors.push(`${item.name}: ${getInputLabel(inputType)} is required`);
            }
        });
    });
    
    return errors;
}

function displayValidationErrors(errors) {
    const summary = document.getElementById('validationSummary');
    const list = document.getElementById('validationList');
    
    if (errors.length > 0) {
        list.innerHTML = errors.map(err => `<li>${err}</li>`).join('');
        summary.classList.remove('hidden');
        return false;
    } else {
        summary.classList.add('hidden');
        return true;
    }
}

// ==================
// CALCULATION
// ==================

function calculateTotals() {
    // Validate first
    const errors = validateInputs();
    if (!displayValidationErrors(errors)) {
        // Scroll to validation summary
        document.getElementById('validationSummary').scrollIntoView({ behavior: 'smooth' });
        return;
    }
    
    // Calculate category totals and base totals
    const categoryTotals = {};
    let grandTotal = 0;
    let baseGrandTotal = 0; // Always calculate base total in background
    
    state.selectedItems.forEach(itemId => {
        const item = ITEMS_DATA.find(i => i.id === itemId);
        const calc = state.calculations[itemId];
        
        if (calc && calc.lineTotal > 0) {
            if (!categoryTotals[item.category]) {
                categoryTotals[item.category] = 0;
            }
            categoryTotals[item.category] += calc.lineTotal;
            grandTotal += calc.lineTotal;
            
            // ALWAYS calculate base total (even for sales consultants - just hidden)
            const basePrice = item.basePrice || BASE_PRICE_MAP[item.id] || 0;
            const baseLineTotal = calc.adjustedQty * basePrice;
            baseGrandTotal += baseLineTotal;
        }
    });
    
    // Store totals in state for later use (email, etc.)
    state.grandTotal = grandTotal;
    state.baseGrandTotal = baseGrandTotal;
    
    // Display category totals
    const categoryTotalsContainer = document.getElementById('categoryTotals');
    categoryTotalsContainer.innerHTML = Object.entries(categoryTotals).map(([category, total]) => `
        <div class="category-total">
            <div class="category-total-label">${category}</div>
            <div class="category-total-amount">${formatCurrency(total)}</div>
        </div>
    `).join('');
    
    // Display grand total
    document.getElementById('grandTotal').textContent = formatCurrency(grandTotal);
    
    // Display BASE GRAND TOTAL only for pricing managers
    console.log('🔍 Checking BASE GRAND TOTAL display:', {
        isPricingManager: IS_PRICING_MANAGER,
        baseGrandTotal: baseGrandTotal,
        grandTotal: grandTotal
    });
    
    let baseGrandTotalElement = document.getElementById('baseGrandTotal');
    if (IS_PRICING_MANAGER) {
        console.log('✅ Pricing Manager - Showing BASE GRAND TOTAL');
        if (!baseGrandTotalElement) {
            // Create wrapper for both totals side by side
            const grandTotalDiv = document.querySelector('.grand-total');
            if (grandTotalDiv) {
                // Create a wrapper for both totals
                const totalsWrapper = document.createElement('div');
                totalsWrapper.id = 'totalsWrapper';
                totalsWrapper.style.display = 'flex';
                totalsWrapper.style.gap = '20px';
                totalsWrapper.style.justifyContent = 'space-between';
                
                // Move existing grand total into the wrapper
                const sellingDiv = document.createElement('div');
                sellingDiv.className = 'grand-total';
                sellingDiv.style.flex = '1';
                sellingDiv.innerHTML = grandTotalDiv.innerHTML;
                
                // Create base grand total
                const baseDiv = document.createElement('div');
                baseDiv.className = 'grand-total base-grand-total-display';
                baseDiv.id = 'baseGrandTotalContainer';
                baseDiv.style.flex = '1';
                baseDiv.style.background = 'linear-gradient(135deg, #e67e22, #d35400)';
                baseDiv.innerHTML = `
                    <span class="total-label" data-en="BASE GRAND TOTAL:" data-ar="المجموع الإجمالي الأساسي:">BASE GRAND TOTAL:</span>
                    <span class="total-amount" id="baseGrandTotal">${formatCurrency(baseGrandTotal)}</span>
                `;
                
                // Add both to wrapper
                totalsWrapper.appendChild(sellingDiv);
                totalsWrapper.appendChild(baseDiv);
                
                // Replace original grand total with wrapper
                grandTotalDiv.parentNode.replaceChild(totalsWrapper, grandTotalDiv);
                
                console.log('✅ BASE GRAND TOTAL element created side by side');
            } else {
                console.error('❌ Could not find .grand-total div');
            }
        } else {
            baseGrandTotalElement.textContent = formatCurrency(baseGrandTotal);
            console.log('✅ BASE GRAND TOTAL updated:', formatCurrency(baseGrandTotal));
        }
    } else {
        console.log('👤 Sales Consultant - Hiding BASE GRAND TOTAL');
        // For sales consultants, if wrapper exists, restore original grand total
        const totalsWrapper = document.getElementById('totalsWrapper');
        if (totalsWrapper) {
            const grandTotalDiv = totalsWrapper.querySelector('.grand-total:first-child');
            if (grandTotalDiv) {
                // Create a standalone grand total div
                const newGrandTotal = document.createElement('div');
                newGrandTotal.className = 'grand-total';
                newGrandTotal.innerHTML = grandTotalDiv.innerHTML;
                totalsWrapper.parentNode.replaceChild(newGrandTotal, totalsWrapper);
                console.log('✅ Restored single GRAND TOTAL for sales');
            }
        }
    }
}

// ==================
// ACTIONS
// ==================

function clearAll() {
    if (!confirm('Are you sure you want to clear all data? This cannot be undone.')) {
        return;
    }
    
    // Clear header fields
    document.getElementById('customerName').value = '';
    document.getElementById('siteLocation').value = '';
    document.getElementById('numRooms').value = '';
    document.getElementById('avgRoomArea').value = '';
    document.getElementById('paintColorCode').value = '';
    document.getElementById('generalNotes').value = '';
    
    // Uncheck all items
    state.selectedItems.forEach(itemId => {
        const checkbox = document.getElementById(`checkbox-${itemId}`);
        if (checkbox) checkbox.checked = false;
        toggleItem(itemId);
    });
    
    // Clear state
    state.selectedItems.clear();
    state.itemInputs = {};
    state.calculations = {};
    
    // Clear totals
    document.getElementById('categoryTotals').innerHTML = '';
    document.getElementById('grandTotal').textContent = '0 IQD';
    document.getElementById('validationSummary').classList.add('hidden');
}

async function printCalculator() {
    // Calculate first if there are selected items
    if (state.selectedItems.size > 0) {
        const errors = validateInputs();
        if (errors.length > 0) {
            return;
        }
        calculateTotals();
        
        // Create print-friendly items list
        createPrintItemsList();
        
        // Send order data to server for internal PDF email (both roles)
        await sendOrderToServer();
    }
    
    // Show customer PDF on screen
    window.print();
    
    // Clean up after print
    setTimeout(() => {
        const printItems = document.getElementById('print-items-list');
        if (printItems) {
            printItems.remove();
        }
    }, 1000);
}

async function sendOrderToServer() {
    console.log('📧 Starting email send process...');
    console.log('User Role:', USER_ROLE, 'Is Pricing Manager:', IS_PRICING_MANAGER);
    
    try {
        // Generate order number
        const orderNumber = generateOrderNumber();
        console.log('📋 Order Number:', orderNumber);
        
        // Collect order data
        const orderData = {
            orderNumber: orderNumber,
            date: new Date().toLocaleString('en-GB', { 
                year: 'numeric', 
                month: '2-digit', 
                day: '2-digit',
                hour: '2-digit',
                minute: '2-digit',
                hour12: false
            }),
            customerName: document.getElementById('customerName').value.trim(),
            siteLocation: document.getElementById('siteLocation').value.trim(),
            generalNotes: document.getElementById('generalNotes').value.trim(),
            items: [],
            sellingGrandTotal: state.grandTotal || 0,
            baseGrandTotal: state.baseGrandTotal || 0
        };
        
        // Collect all selected items with their calculations
        state.selectedItems.forEach(itemId => {
            const item = ITEMS_DATA.find(i => i.id === itemId);
            const calc = state.calculations[itemId];
            const inputs = state.itemInputs[itemId];
            
            if (calc && calc.adjustedQty > 0) {
                // Get current selling price (may have been edited)
                let currentPrice = item.price;
                if (IS_PRICING_MANAGER) {
                    const priceInput = document.querySelector(`.price-input[data-item-id="${itemId}"]`);
                    if (priceInput) {
                        const inputPrice = parseFloat(priceInput.value);
                        if (!isNaN(inputPrice) && inputPrice >= 0) {
                            currentPrice = inputPrice;
                        }
                    }
                }
                
                // Get base price
                const basePrice = item.basePrice || BASE_PRICE_MAP[item.id] || 0;
                const baseLineTotal = calc.adjustedQty * basePrice;
                
                // Format inputs display
                let inputsDisplay = '';
                if (inputs.length) inputsDisplay += `L: ${inputs.length}m `;
                if (inputs.width) inputsDisplay += `W: ${inputs.width}m `;
                if (inputs.quantity) inputsDisplay += `Qty: ${inputs.quantity} `;
                if (inputs.rooms) inputsDisplay += `Rooms: ${inputs.rooms} `;
                if (inputs.avgArea) inputsDisplay += `Avg: ${inputs.avgArea}m² `;
                
                orderData.items.push({
                    category: item.category,
                    name: item.name,
                    unit: item.unit,
                    inputs: inputsDisplay.trim(),
                    adjustedQty: calc.adjustedQty % 1 === 0 ? 
                        calc.adjustedQty.toFixed(0) : 
                        parseFloat(calc.adjustedQty.toFixed(2)).toString(),
                    sellingPrice: currentPrice,
                    sellingLineTotal: calc.lineTotal,
                    basePrice: basePrice,
                    baseLineTotal: baseLineTotal
                });
            }
        });
        
        // Send to server
        const response = await fetch('send_order_email.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(orderData)
        });
        
        // Read response as text first to see what we got
        const responseText = await response.text();
        
        if (!response.ok) {
            console.error('❌ HTTP Error:', responseText);
            throw new Error(`HTTP ${response.status}: ${responseText}`);
        }
        
        // Try to parse as JSON
        let result;
        try {
            result = JSON.parse(responseText);
        } catch (e) {
            console.error('❌ JSON Parse Error:', e.message);
            console.error('Response was:', responseText);
            throw new Error('Server returned invalid JSON: ' + responseText.substring(0, 200));
        }
        
        if (result.success) {
        } else {
            console.warn('⚠️ Email sending failed:', result.message);
            alert('⚠️ Warning: Order processed but email may not have been sent. Check with administrator.');
        }
    } catch (error) {
        console.error('❌ Error sending order:', error);
        console.error('Error details:', error.message, error.stack);
        alert('⚠️ Warning: Could not send email notification. The PDF will still be generated.');
        // Don't block printing if email fails
    }
}

function generateOrderNumber() {
    const now = new Date();
    const year = now.getFullYear();
    const month = String(now.getMonth() + 1).padStart(2, '0');
    const day = String(now.getDate()).padStart(2, '0');
    const hours = String(now.getHours()).padStart(2, '0');
    const minutes = String(now.getMinutes()).padStart(2, '0');
    const seconds = String(now.getSeconds()).padStart(2, '0');
    const random = Math.floor(Math.random() * 1000).toString().padStart(3, '0');
    
    return `${year}${month}${day}-${hours}${minutes}${seconds}-${random}`;
}

function createPrintItemsList() {
    // Remove existing print list if any
    const existing = document.getElementById('print-items-list');
    if (existing) existing.remove();
    
    // Create items table for print
    const printDiv = document.createElement('div');
    printDiv.id = 'print-items-list';
    printDiv.className = 'print-only';
    
    let itemsHTML = '<table class="print-items-table">';
    itemsHTML += '<thead><tr>';
    itemsHTML += '<th>Item</th>';
    itemsHTML += '<th>Unit</th>';
    itemsHTML += '<th>Inputs</th>';
    itemsHTML += '<th>Adj. Qty</th>';
    
    // For pricing managers, show both selling and base prices
    if (IS_PRICING_MANAGER) {
        itemsHTML += '<th>Selling Price</th>';
        itemsHTML += '<th>Base Price</th>';
        itemsHTML += '<th>Selling Total</th>';
        itemsHTML += '<th>Base Total</th>';
    } else {
        itemsHTML += '<th>Unit Price</th>';
        itemsHTML += '<th>Line Total</th>';
    }
    
    itemsHTML += '</tr></thead><tbody>';
    
    state.selectedItems.forEach(itemId => {
        const item = ITEMS_DATA.find(i => i.id === itemId);
        const calc = state.calculations[itemId];
        const inputs = state.itemInputs[itemId];
        
        if (calc && calc.adjustedQty > 0) {
            // Format inputs display
            let inputsDisplay = '';
            if (inputs.length) inputsDisplay += `L: ${inputs.length}m `;
            if (inputs.width) inputsDisplay += `W: ${inputs.width}m `;
            if (inputs.quantity) inputsDisplay += `Qty: ${inputs.quantity} `;
            if (inputs.rooms) inputsDisplay += `Rooms: ${inputs.rooms} `;
            if (inputs.avgArea) inputsDisplay += `Avg: ${inputs.avgArea}m² `;
            
            const displayQty = calc.adjustedQty % 1 === 0 ? 
                calc.adjustedQty.toFixed(0) : 
                parseFloat(calc.adjustedQty.toFixed(2)).toString();
            
            // Get current price (may have been edited)
            let currentPrice = item.price;
            if (IS_PRICING_MANAGER) {
                const priceInput = document.querySelector(`.price-input[data-item-id="${itemId}"]`);
                if (priceInput) {
                    const inputPrice = parseFloat(priceInput.value);
                    if (!isNaN(inputPrice) && inputPrice >= 0) {
                        currentPrice = inputPrice;
                    }
                }
            }
            
            // Get base price and calculate base line total
            const basePrice = item.basePrice || BASE_PRICE_MAP[item.id] || 0;
            const baseLineTotal = calc.adjustedQty * basePrice;
            
            itemsHTML += '<tr>';
            itemsHTML += `<td class="item-name-cell">${item.name}</td>`;
            itemsHTML += `<td>${item.unit}</td>`;
            itemsHTML += `<td>${inputsDisplay}</td>`;
            itemsHTML += `<td class="qty-cell">${displayQty} ${item.unit}</td>`;
            
            // For pricing managers, show both prices and both totals
            if (IS_PRICING_MANAGER) {
                itemsHTML += `<td class="price-cell selling-price">${formatCurrency(currentPrice)}</td>`;
                itemsHTML += `<td class="price-cell base-price">${formatCurrency(basePrice)}</td>`;
                itemsHTML += `<td class="total-cell selling-total">${formatCurrency(calc.lineTotal)}</td>`;
                itemsHTML += `<td class="total-cell base-total">${formatCurrency(baseLineTotal)}</td>`;
            } else {
                itemsHTML += `<td class="price-cell">${formatCurrency(currentPrice)}</td>`;
                itemsHTML += `<td class="total-cell">${formatCurrency(calc.lineTotal)}</td>`;
            }
            
            itemsHTML += '</tr>';
        }
    });
    
    itemsHTML += '</tbody></table>';
    printDiv.innerHTML = itemsHTML;
    
    // Insert before summary section
    const summarySection = document.querySelector('.summary-section');
    summarySection.insertBefore(printDiv, summarySection.firstChild);
}

// ==================
// LANGUAGE TOGGLE
// ==================

let currentLanguage = 'en';

function toggleLanguage() {
    currentLanguage = currentLanguage === 'en' ? 'ar' : 'en';
    
    // Update all elements with data-en and data-ar attributes
    document.querySelectorAll('[data-en]').forEach(element => {
        const text = element.getAttribute(`data-${currentLanguage}`);
        if (text) {
            element.textContent = text;
        }
    });
    
    // Update placeholders
    document.querySelectorAll('[data-placeholder-en]').forEach(element => {
        const placeholder = element.getAttribute(`data-placeholder-${currentLanguage}`);
        if (placeholder) {
            element.placeholder = placeholder;
        }
    });
    
    // Update language toggle button
    document.getElementById('langText').textContent = currentLanguage === 'en' ? 'العربية' : 'English';
    
    // Update category names and item names
    updateCategoryTranslations();
    
    console.log(`Language switched to: ${currentLanguage}`);
}

function updateCategoryTranslations() {
    // Category translations
    const categoryTranslations = {
        'Gypsum Board Items': 'ألواح الجبس',
        'Gypsum Decorative & Lighting': 'الجبس الديكوري والإضاءة',
        'Spot Lights': 'الأضواء النقطية',
        'Profile LED (Standard)': 'بروفايل LED (عادي)',
        'Profile LED (Black)': 'بروفايل LED (أسود)',
        'Profile Magnetic': 'بروفايل مغناطيسي',
        'Power Supplies & Magnetic LED': 'مصادر الطاقة وLED المغناطيسي',
        'Wall Coverings': 'تغطيات الجدران',
        'Paint Items': 'عناصر الطلاء',
        'Electricity': 'الكهرباء',
        'Gypsum Decorative Panels (Flat)': 'ألواح جبس ديكور (مسطحة)',
        'Gypsum Decorative Panels (Decor)': 'ألواح جبس ديكور (مزخرفة)',
        'Gypsum Decorative Panels (Ceiling)': 'ألواح جبس ديكور (سقف)',
        'Gypsum Decorative Panels (Angle)': 'ألواح جبس ديكور (زاوية)',
        'Gypsum Decorative Panels (Royal)': 'ألواح جبس ديكور (ملكي)',
        'Miscellaneous': 'متفرقات',
        'Gypsum Board Decor': 'ألواح جبس ديكور'
    };
    
    // Update rendered categories
    const categories = document.querySelectorAll('.category-title span:last-child');
    categories.forEach(categoryElement => {
        const englishName = categoryElement.getAttribute('data-category-en');
        if (englishName && categoryTranslations[englishName]) {
            if (currentLanguage === 'ar') {
                categoryElement.textContent = categoryTranslations[englishName];
            } else {
                categoryElement.textContent = englishName;
            }
        }
    });
}

// ==================
// INITIALIZATION
// ==================

document.addEventListener('DOMContentLoaded', () => {
    // Load dynamic prices from PHP
    loadDynamicPrices();
    
    renderCategories();
    
    // Attach event listeners
    document.getElementById('calculateBtn').addEventListener('click', calculateTotals);
    document.getElementById('clearBtn').addEventListener('click', clearAll);
    document.getElementById('printBtn').addEventListener('click', printCalculator);
    
    // Update conditional requirements on header field changes
    ['numRooms', 'avgRoomArea', 'paintColorCode'].forEach(fieldId => {
        document.getElementById(fieldId).addEventListener('input', updateConditionalRequirements);
    });
    
    // Add event listeners for price changes (pricing managers only)
    if (IS_PRICING_MANAGER) {
        document.addEventListener('input', (e) => {
            if (e.target.classList.contains('price-input')) {
                const itemId = parseInt(e.target.dataset.itemId);
                if (state.selectedItems.has(itemId)) {
                    updateItemCalculation(itemId);
                }
            }
        });
    }
    
    console.log('================================================');
    console.log('✅ Bait Alamer Customer Portal Calculator initialized with 57 items');
    console.log('👤 User Role:', USER_ROLE);
    console.log('🔒 Pricing Manager:', IS_PRICING_MANAGER);
    console.log('💰 Base Prices Loaded:', Object.keys(BASE_PRICE_MAP).length);
    console.log('💵 Selling Prices Loaded:', Object.keys(SELLING_PRICE_MAP).length);
    console.log('🖨️ Use "Print / Export PDF" button to generate professional quotation');
    console.log('================================================');
});

