<?php
session_start();
require_once 'users.php';
require_once 'pricing_functions.php';

if (isSystemLocked()) {
    header('Location: maintenance.php');
    exit;
}

// Check if user is logged in
if (!isLoggedIn()) {
    header('Location: login.php');
    exit;
}

// Load base prices from the single source of truth
require_once '../load_base_prices.php';
$basePriceMap = loadBasePrices();

// Load selling prices (customer's prices)
$sellingPricesFile = 'selling_prices.json';
$sellingPriceMap = [];

if (file_exists($sellingPricesFile)) {
    $sellingPricesJson = file_get_contents($sellingPricesFile);
    $sellingPrices = json_decode($sellingPricesJson, true);
    
    if (is_array($sellingPrices)) {
        foreach ($sellingPrices as $item) {
            if (isset($item['id']) && isset($item['selling_price'])) {
                $sellingPriceMap[$item['id']] = $item['selling_price'];
            }
        }
    }
}

// If no selling prices exist yet, initialize from base prices
if (empty($sellingPriceMap)) {
    $sellingPriceMap = $basePriceMap;
}

$username = $_SESSION['username'];
$role = $_SESSION['role'];
$isPricingManager = hasRole('pricing_manager');

// Load announcements for pricing managers
$activeAnnouncements = [];
if ($isPricingManager) {
    $announcementsFile = '../announcements.json';
    if (file_exists($announcementsFile)) {
        $announcementsJson = file_get_contents($announcementsFile);
        $allAnnouncements = json_decode($announcementsJson, true) ?: [];
        // Filter only active announcements
        $activeAnnouncements = array_filter($allAnnouncements, function($a) {
            return isset($a['active']) && $a['active'] === true;
        });
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Bait Alamer Pricing Calculator - Customer Portal</title>
    <link rel="stylesheet" href="styles.css">
    <link rel="stylesheet" href="print-styles.css" media="print">
    <style>
        @keyframes slideInFromTop {
            from {
                opacity: 0;
                transform: translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        @keyframes fadeInScale {
            from {
                opacity: 0;
                transform: scale(0.95);
            }
            to {
                opacity: 1;
                transform: scale(1);
            }
        }
        
        .user-info-bar {
            background: linear-gradient(135deg, #2c3e50, #34495e);
            color: white;
            padding: 15px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            border-radius: 8px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
            animation: slideInFromTop 0.6s ease-out;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .user-info-bar::before {
            content: '👤';
            position: absolute;
            right: 10px;
            top: 50%;
            transform: translateY(-50%);
            font-size: 50px;
            opacity: 0.05;
            pointer-events: none;
        }
        
        .user-info-bar:hover {
            box-shadow: 0 6px 20px rgba(0,0,0,0.3);
            transform: translateY(-2px);
        }
        
        .user-info-bar .user-details {
            display: flex;
            align-items: center;
            gap: 20px;
            position: relative;
            z-index: 1;
        }
        
        .user-info-bar .role-badge {
            background: linear-gradient(135deg, #e67e22, #d35400);
            padding: 6px 18px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 700;
            text-transform: uppercase;
            box-shadow: 0 2px 8px rgba(230, 126, 34, 0.3);
            transition: all 0.3s cubic-bezier(0.68, -0.55, 0.265, 1.55);
            letter-spacing: 0.5px;
        }
        
        .user-info-bar .role-badge:hover {
            transform: scale(1.05) translateY(-2px);
            box-shadow: 0 4px 12px rgba(230, 126, 34, 0.5);
        }
        
        .user-info-bar .logout-btn {
            background: linear-gradient(135deg, #c0392b, #e74c3c);
            color: white;
            border: none;
            padding: 10px 24px;
            border-radius: 6px;
            cursor: pointer;
            font-weight: 700;
            text-decoration: none;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            box-shadow: 0 2px 8px rgba(192, 57, 43, 0.3);
            position: relative;
            overflow: hidden;
        }
        
        .user-info-bar .logout-btn::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 0;
            height: 0;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.2);
            transform: translate(-50%, -50%);
            transition: width 0.4s ease, height 0.4s ease;
        }
        
        .user-info-bar .logout-btn:hover::before {
            width: 300px;
            height: 300px;
        }
        
        .user-info-bar .logout-btn:hover {
            background: linear-gradient(135deg, #e74c3c, #c0392b);
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(231, 76, 60, 0.5);
        }
        
        .user-info-bar .help-btn {
            background: linear-gradient(135deg, #25D366, #128C7E);
            color: white;
            border: none;
            padding: 10px 24px;
            border-radius: 6px;
            cursor: pointer;
            font-weight: 700;
            text-decoration: none;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            box-shadow: 0 2px 8px rgba(37, 211, 102, 0.3);
            position: relative;
            overflow: hidden;
            margin-right: 10px;
        }
        
        .user-info-bar .help-btn::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 0;
            height: 0;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.2);
            transform: translate(-50%, -50%);
            transition: width 0.4s ease, height 0.4s ease;
        }
        
        .user-info-bar .help-btn:hover::before {
            width: 300px;
            height: 300px;
        }
        
        .user-info-bar .help-btn:hover {
            background: linear-gradient(135deg, #128C7E, #25D366);
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(37, 211, 102, 0.5);
        }
        
        .price-input {
            width: 110px;
            padding: 6px 10px;
            border: 2px solid #e67e22;
            border-radius: 6px;
            text-align: right;
            font-weight: bold;
            font-size: 13px;
            transition: all 0.3s ease;
            background: white;
            position: relative;
            z-index: 10;
            cursor: text;
        }
        
        .price-input:focus {
            outline: none;
            border-color: #f39c12;
            box-shadow: 0 0 0 3px rgba(243, 156, 18, 0.2);
            transform: scale(1.02);
            z-index: 20;
        }
        
        .price-display {
            font-weight: bold;
            color: #fff;
        }
        
        .save-prices-section {
            background: linear-gradient(135deg, #fff3cd, #ffe69c);
            border: 2px solid #ffc107;
            padding: 25px;
            border-radius: 10px;
            margin-bottom: 20px;
            text-align: center;
            box-shadow: 0 4px 15px rgba(255, 193, 7, 0.2);
            animation: fadeInScale 0.6s ease-out 0.2s backwards;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .save-prices-section::before {
            content: '💾';
            position: absolute;
            right: 20px;
            top: 50%;
            transform: translateY(-50%);
            font-size: 60px;
            opacity: 0.1;
            pointer-events: none;
        }
        
        .save-prices-section:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 25px rgba(255, 193, 7, 0.3);
        }
        
        .save-prices-btn {
            background: linear-gradient(135deg, #28a745, #20c997);
            color: white;
            border: none;
            padding: 14px 35px;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            box-shadow: 0 4px 12px rgba(40, 167, 69, 0.3);
            text-transform: uppercase;
            letter-spacing: 1px;
            position: relative;
            overflow: hidden;
        }
        
        .save-prices-btn::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 0;
            height: 0;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.2);
            transform: translate(-50%, -50%);
            transition: width 0.5s ease, height 0.5s ease;
        }
        
        .save-prices-btn:hover::before {
            width: 300px;
            height: 300px;
        }
        
        .save-prices-btn:hover {
            background: linear-gradient(135deg, #20c997, #28a745);
            transform: translateY(-3px) scale(1.02);
            box-shadow: 0 6px 20px rgba(40, 167, 69, 0.5);
        }
        
        .save-prices-btn:active {
            transform: translateY(-1px) scale(0.98);
        }
        
        .price-cell {
            min-width: 150px;
        }
        
        .announcements-section {
            background: linear-gradient(135deg, #3498db, #2980b9);
            color: white;
            padding: 25px;
            border-radius: 10px;
            margin-bottom: 20px;
            box-shadow: 0 4px 20px rgba(52, 152, 219, 0.3);
            animation: fadeInScale 0.6s ease-out 0.1s backwards;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .announcements-section::before {
            content: '📢';
            position: absolute;
            right: 15px;
            top: 15px;
            font-size: 50px;
            opacity: 0.1;
            pointer-events: none;
            animation: pulse 2s ease-in-out infinite;
        }
        
        @keyframes pulse {
            0%, 100% { transform: scale(1); opacity: 0.1; }
            50% { transform: scale(1.1); opacity: 0.15; }
        }
        
        .announcements-section:hover {
            box-shadow: 0 6px 30px rgba(52, 152, 219, 0.4);
            transform: translateY(-2px);
        }
        
        .announcements-section h3 {
            margin: 0 0 18px 0;
            font-size: 20px;
            font-weight: 700;
            letter-spacing: 0.5px;
            position: relative;
            z-index: 1;
        }
        
        .announcement-item {
            background: rgba(255,255,255,0.15);
            padding: 18px;
            border-radius: 8px;
            margin-bottom: 12px;
            border-left: 4px solid white;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            backdrop-filter: blur(5px);
            animation: fadeInScale 0.4s ease-out backwards;
        }
        
        .announcement-item:nth-child(1) { animation-delay: 0.3s; }
        .announcement-item:nth-child(2) { animation-delay: 0.4s; }
        .announcement-item:nth-child(3) { animation-delay: 0.5s; }
        
        .announcement-item:hover {
            background: rgba(255,255,255,0.25);
            transform: translateX(5px);
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
        }
        
        .announcement-item:last-child {
            margin-bottom: 0;
        }
        
        .announcement-message {
            margin: 0 0 8px 0;
            font-size: 15px;
            font-weight: 500;
            line-height: 1.5;
        }
        
        .announcement-date {
            margin: 0;
            font-size: 12px;
            opacity: 0.9;
            font-weight: 600;
        }
        
        .user-management-toggle-btn {
            background: linear-gradient(135deg, #e67e22, #d35400);
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.3s ease;
            box-shadow: 0 2px 8px rgba(230, 126, 34, 0.3);
            margin-bottom: 20px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .user-management-toggle-btn:hover {
            background: linear-gradient(135deg, #d35400, #e67e22);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(230, 126, 34, 0.5);
        }
        
        .user-management-section {
            background: linear-gradient(135deg, #fff3cd, #ffe69c);
            border: 2px solid #ffc107;
            color: #000;
            padding: 25px;
            border-radius: 10px;
            margin-bottom: 20px;
            box-shadow: 0 4px 15px rgba(255, 193, 7, 0.2);
            animation: fadeInScale 0.6s ease-out;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
            display: none;
        }
        
        .user-management-section.show {
            display: block;
        }
        
        .user-management-section::before {
            content: '👥';
            position: absolute;
            right: 20px;
            top: 20px;
            font-size: 60px;
            opacity: 0.1;
            pointer-events: none;
        }
        
        .user-management-section:hover {
            box-shadow: 0 6px 25px rgba(255, 193, 7, 0.3);
            transform: translateY(-2px);
        }
        
        .user-management-section h3 {
            margin: 0 0 20px 0;
            font-size: 20px;
            font-weight: 700;
            letter-spacing: 0.5px;
            position: relative;
            z-index: 1;
            color: #000;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .close-btn {
            background: #c0392b;
            color: white;
            border: none;
            width: 30px;
            height: 30px;
            border-radius: 50%;
            font-size: 18px;
            font-weight: bold;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
            box-shadow: 0 2px 6px rgba(192, 57, 43, 0.3);
        }
        
        .close-btn:hover {
            background: #e74c3c;
            transform: scale(1.1) rotate(90deg);
            box-shadow: 0 4px 10px rgba(192, 57, 43, 0.5);
        }
        
        .user-password-form {
            background: rgba(255,255,255,0.7);
            padding: 18px;
            border-radius: 8px;
            margin-bottom: 12px;
            border-left: 4px solid #ffc107;
            transition: all 0.3s ease;
        }
        
        .user-password-form:hover {
            background: rgba(255,255,255,0.9);
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        
        .user-password-form:last-child {
            margin-bottom: 0;
        }
        
        .user-password-form label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            font-size: 14px;
            color: #333;
        }
        
        .user-password-form input {
            width: 100%;
            padding: 10px;
            border: 2px solid #ffc107;
            border-radius: 6px;
            background: white;
            color: #333;
            font-size: 14px;
            margin-bottom: 10px;
            transition: all 0.3s ease;
        }
        
        .user-password-form input:focus {
            outline: none;
            border-color: #e67e22;
            box-shadow: 0 0 0 3px rgba(230, 126, 34, 0.2);
        }
        
        .change-password-btn {
            background: linear-gradient(135deg, #e67e22, #d35400);
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            box-shadow: 0 2px 8px rgba(230, 126, 34, 0.3);
        }
        
        .change-password-btn:hover {
            background: linear-gradient(135deg, #d35400, #e67e22);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(230, 126, 34, 0.5);
        }
        
        .change-password-btn:active {
            transform: translateY(0);
        }
        
        .change-password-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        .password-message {
            margin-top: 10px;
            padding: 8px;
            border-radius: 5px;
            font-size: 13px;
            display: none;
        }
        
        .password-message.success {
            background: rgba(40, 167, 69, 0.2);
            color: #28a745;
            border: 1px solid rgba(40, 167, 69, 0.5);
        }
        
        .password-message.error {
            background: rgba(220, 53, 69, 0.2);
            color: #c33;
            border: 1px solid rgba(220, 53, 69, 0.5);
        }
        
        .self-password-form {
            background: rgba(255,255,255,0.7);
            padding: 18px;
            border-radius: 8px;
            margin-bottom: 12px;
            border-left: 4px solid #e67e22;
            transition: all 0.3s ease;
        }
        
        .self-password-form:hover {
            background: rgba(255,255,255,0.9);
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        
        .self-password-form label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            font-size: 14px;
            color: #333;
        }
        
        .self-password-form input {
            width: 100%;
            padding: 10px;
            border: 2px solid #e67e22;
            border-radius: 6px;
            background: white;
            color: #333;
            font-size: 14px;
            margin-bottom: 10px;
            transition: all 0.3s ease;
        }
        
        .self-password-form input:focus {
            outline: none;
            border-color: #d35400;
            box-shadow: 0 0 0 3px rgba(211, 84, 0, 0.2);
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- User Info Bar -->
        <div class="user-info-bar no-print">
            <div class="user-details">
                <span>👤 <strong><?php echo htmlspecialchars(ucfirst($username)); ?></strong></span>
                <span class="role-badge"><?php echo $isPricingManager ? 'Pricing Manager' : 'Sales Consultant'; ?></span>
            </div>
            <div style="display: flex; align-items: center; gap: 10px;">
                <a href="https://wa.me/9647724172085" target="_blank" class="help-btn">💬 Help</a>
                <a href="logout.php" class="logout-btn">Logout</a>
            </div>
        </div>

        <?php if ($isPricingManager): ?>
        <!-- Announcements Section (Only for Pricing Managers) -->
        <?php if (!empty($activeAnnouncements)): ?>
        <div class="announcements-section no-print">
            <h3>📢 Important Announcements</h3>
            <?php foreach ($activeAnnouncements as $announcement): ?>
                <div class="announcement-item">
                    <p class="announcement-message"><?php echo nl2br(htmlspecialchars($announcement['message'])); ?></p>
                    <p class="announcement-date">📅 <?php echo htmlspecialchars($announcement['date']); ?></p>
                </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>

        <!-- Save Prices Section (Only for Pricing Managers) -->
        <div class="save-prices-section no-print">
            <p style="margin: 0 0 10px 0; font-weight: 600; color: #000;">💡 You can edit unit prices below. Base prices shown for reference. Click "Save Prices" when done.</p>
            <button type="button" class="save-prices-btn" onclick="savePrices()">💾 Save Prices</button>
        </div>

        <!-- User Management Toggle Button (Only for Pricing Managers) -->
        <button type="button" class="user-management-toggle-btn no-print" onclick="toggleUserManagement()">
            👥 Manage User Passwords
        </button>

        <!-- User Management Section (Only for Pricing Managers) -->
        <div class="user-management-section no-print" id="userManagementSection">
            <h3>
                <span>👥 User Password Management</span>
                <button type="button" class="close-btn" onclick="toggleUserManagement()" title="Close">×</button>
            </h3>
            
            <!-- Self Password Change Form -->
            <div class="self-password-form">
                <label>
                    Change your own password: <strong><?php 
                        $displayUsername = ucfirst($username);
                        echo htmlspecialchars($displayUsername); 
                    ?></strong>
                </label>
                <input 
                    type="password" 
                    id="old-password-self" 
                    placeholder="Enter your current password"
                >
                <input 
                    type="password" 
                    id="new-password-self" 
                    placeholder="Enter new password (minimum 6 characters)"
                    minlength="6"
                >
                <button 
                    type="button" 
                    class="change-password-btn" 
                    onclick="changeSelfPassword()"
                >
                    🔒 Change My Password
                </button>
                <div 
                    class="password-message" 
                    id="message-self"
                ></div>
            </div>
            
            <?php
            require_once 'users.php';
            $allUsers = getAllUsers();
            // Filter to show only sales user
            $usersToManage = array_filter($allUsers, function($user) {
                return $user['username'] === 'sales';
            });
            foreach ($usersToManage as $user):
                $displayUsername = ucfirst($user['username']);
            ?>
            <div class="user-password-form">
                <label for="password-<?php echo htmlspecialchars($user['username']); ?>">
                    Change password for: <strong><?php echo htmlspecialchars($displayUsername); ?></strong>
                </label>
                <input 
                    type="password" 
                    id="password-<?php echo htmlspecialchars($user['username']); ?>" 
                    placeholder="Enter new password (minimum 6 characters)"
                    minlength="6"
                >
                <button 
                    type="button" 
                    class="change-password-btn" 
                    onclick="changePassword('<?php echo htmlspecialchars($user['username']); ?>')"
                >
                    🔒 Change Password
                </button>
                <div 
                    class="password-message" 
                    id="message-<?php echo htmlspecialchars($user['username']); ?>"
                ></div>
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>

        <!-- Header -->
        <header class="page-header">
            <div class="language-toggle">
                <button id="langToggle" onclick="toggleLanguage()">
                    <span id="langText">العربية</span>
                </button>
            </div>
            <div class="header-content">
                <div class="header-logo">
                    <img src="logo.png" alt="Bait Alamer Logo">
                </div>
                <div class="header-text">
                    <h1 data-en="Bait Alamer Pricing Calculator" data-ar="حاسبة التسعير - بيت العامر">Bait Alamer Pricing Calculator</h1>
                    <p class="subtitle" data-en="Professional Quality Assurance Pricing Tool" data-ar="أداة تسعير احترافية لضمان الجودة">Professional Quality Assurance Pricing Tool</p>
                </div>
            </div>
        </header>

        <!-- Section A: Header Fields -->
        <section class="header-fields">
            <h2 data-en="Site Information" data-ar="معلومات الموقع">Site Information</h2>
            <div class="field-grid">
                <div class="field-group">
                    <label for="customerName">
                        <span class="required">*</span> <span data-en="Customer Name" data-ar="اسم العميل">Customer Name</span>
                    </label>
                    <input type="text" id="customerName" data-placeholder-en="Enter customer name" data-placeholder-ar="أدخل اسم العميل" placeholder="Enter customer name">
                    <span class="error-message" id="error-customerName"></span>
                </div>

                <div class="field-group">
                    <label for="siteLocation">
                        <span class="required">*</span> <span data-en="Site Location" data-ar="موقع المشروع">Site Location</span>
                    </label>
                    <input type="text" id="siteLocation" data-placeholder-en="Enter site location" data-placeholder-ar="أدخل موقع المشروع" placeholder="Enter site location">
                    <span class="error-message" id="error-siteLocation"></span>
                </div>

                <div class="field-group">
                    <label for="numRooms">
                        <span class="conditional-required" id="rooms-asterisk">*</span> <span data-en="Number of Rooms" data-ar="عدد الغرف">Number of Rooms</span>
                    </label>
                    <input type="number" id="numRooms" placeholder="0" min="0" step="1">
                    <span class="help-text" data-en="Required if electricity or paint items selected" data-ar="مطلوب عند اختيار عناصر الكهرباء أو الطلاء">Required if electricity or paint items selected</span>
                    <span class="error-message" id="error-numRooms"></span>
                </div>

                <div class="field-group">
                    <label for="avgRoomArea">
                        <span class="conditional-required" id="area-asterisk">*</span> <span data-en="Average Room Area (m²)" data-ar="متوسط مساحة الغرفة (م²)">Average Room Area (m²)</span>
                    </label>
                    <input type="number" id="avgRoomArea" placeholder="0" min="0" step="0.1">
                    <span class="help-text" data-en="Required only if electricity item selected" data-ar="مطلوب فقط عند اختيار عنصر الكهرباء">Required only if electricity item selected</span>
                    <span class="error-message" id="error-avgRoomArea"></span>
                </div>

                <div class="field-group">
                    <label for="paintColorCode">
                        <span class="conditional-required" id="paint-asterisk">*</span> <span data-en="Paint Color Code" data-ar="كود لون الطلاء">Paint Color Code</span>
                    </label>
                    <input type="text" id="paintColorCode" data-placeholder-en="e.g. RAL 9010" data-placeholder-ar="مثال: RAL 9010" placeholder="e.g. RAL 9010">
                    <span class="help-text" data-en="Required only if paint items selected" data-ar="مطلوب فقط عند اختيار عناصر الطلاء">Required only if paint items selected</span>
                    <span class="error-message" id="error-paintColorCode"></span>
                </div>

                <div class="field-group full-width">
                    <label for="generalNotes"><span data-en="General Notes" data-ar="ملاحظات عامة">General Notes</span></label>
                    <textarea id="generalNotes" rows="2" data-placeholder-en="Any additional notes (optional)" data-placeholder-ar="أي ملاحظات إضافية (اختياري)" placeholder="Any additional notes (optional)"></textarea>
                </div>
            </div>
        </section>

        <!-- Section B: Items Table -->
        <section class="items-section">
            <h2 data-en="Select Items & Enter Quantities" data-ar="اختر العناصر وأدخل الكميات">Select Items & Enter Quantities</h2>
            
            <div id="categoriesContainer">
                <!-- Categories will be generated by JavaScript -->
            </div>
        </section>

        <!-- Section C: Summary -->
        <section class="summary-section">
            <h2 data-en="Summary" data-ar="الملخص">Summary</h2>
            
            <div class="category-totals" id="categoryTotals">
                <!-- Category subtotals will be generated here -->
            </div>

            <div class="grand-total">
                <span class="total-label" data-en="GRAND TOTAL:" data-ar="المجموع الإجمالي:">GRAND TOTAL:</span>
                <span class="total-amount" id="grandTotal">0 IQD</span>
            </div>

            <div class="action-buttons">
                <button id="calculateBtn" class="btn btn-primary">
                    <span data-en="📊 Calculate Total" data-ar="📊 احسب المجموع">📊 Calculate Total</span>
                </button>
                <button id="clearBtn" class="btn btn-secondary">
                    <span data-en="🗑️ Clear All" data-ar="🗑️ مسح الكل">🗑️ Clear All</span>
                </button>
                <button id="printBtn" class="btn btn-secondary">
                    <span data-en="🖨️ Print / Export PDF" data-ar="🖨️ طباعة / تصدير PDF">🖨️ Print / Export PDF</span>
                </button>
            </div>

            <div id="validationSummary" class="validation-summary hidden">
                <h3 data-en="⚠️ Please fix the following:" data-ar="⚠️ يرجى إصلاح التالي:">⚠️ Please fix the following:</h3>
                <ul id="validationList"></ul>
            </div>
        </section>

        <footer class="page-footer">
            <p data-en="© 2025 Bait Alamer | Pricing Calculator v1.0 - Customer Portal" data-ar="© 2025 بيت العامر | حاسبة التسعير الإصدار 1.0 - بوابة العملاء">© 2025 Bait Alamer | Pricing Calculator v1.0 - Customer Portal</p>
        </footer>
    </div>

    <script>
        // Inject PHP variables into JavaScript
        const USER_ROLE = '<?php echo $role; ?>';
        const IS_PRICING_MANAGER = <?php echo $isPricingManager ? 'true' : 'false'; ?>;
        const BASE_PRICE_MAP = <?php echo json_encode($basePriceMap); ?>;
        const SELLING_PRICE_MAP = <?php echo json_encode($sellingPriceMap); ?>;
    </script>
    <script src="calculator_customer.js"></script>
    
    <script>
        // Save prices function (only for pricing managers)
        function savePrices() {
            if (!IS_PRICING_MANAGER) {
                alert('You do not have permission to save prices.');
                return;
            }
            
            const priceInputs = document.querySelectorAll('.price-input');
            const updatedPrices = {};
            
            priceInputs.forEach(input => {
                const itemId = parseInt(input.dataset.itemId);
                const newPrice = parseFloat(input.value);
                if (!isNaN(newPrice) && newPrice >= 0) {
                    updatedPrices[itemId] = newPrice;
                }
            });
            
            // Send to server
            fetch('save_prices.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ prices: updatedPrices })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('✅ Prices saved successfully!');
                    location.reload();
                } else {
                    alert('❌ Error: ' + (data.message || 'Failed to save prices'));
                }
            })
            .catch(error => {
                alert('❌ Error saving prices: ' + error.message);
            });
        }
        
        // Toggle user management section
        function toggleUserManagement() {
            const section = document.getElementById('userManagementSection');
            if (section) {
                section.classList.toggle('show');
            }
        }
        
        // Change password function (only for pricing managers) - for other users
        function changePassword(username) {
            if (!IS_PRICING_MANAGER) {
                alert('You do not have permission to change passwords.');
                return;
            }
            
            const passwordInput = document.getElementById('password-' + username);
            const messageDiv = document.getElementById('message-' + username);
            const newPassword = passwordInput.value.trim();
            
            // Validate password
            if (newPassword.length < 6) {
                messageDiv.textContent = '❌ Password must be at least 6 characters long';
                messageDiv.className = 'password-message error';
                messageDiv.style.display = 'block';
                return;
            }
            
            // Disable button during request
            const button = passwordInput.nextElementSibling;
            button.disabled = true;
            button.textContent = '⏳ Updating...';
            
            // Send to server
            fetch('change_password.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    username: username,
                    newPassword: newPassword
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    messageDiv.textContent = '✅ Password updated successfully!';
                    messageDiv.className = 'password-message success';
                    messageDiv.style.display = 'block';
                    passwordInput.value = '';
                    
                    // Hide message after 3 seconds
                    setTimeout(() => {
                        messageDiv.style.display = 'none';
                    }, 3000);
                } else {
                    messageDiv.textContent = '❌ Error: ' + (data.message || 'Failed to update password');
                    messageDiv.className = 'password-message error';
                    messageDiv.style.display = 'block';
                }
            })
            .catch(error => {
                messageDiv.textContent = '❌ Error: ' + error.message;
                messageDiv.className = 'password-message error';
                messageDiv.style.display = 'block';
            })
            .finally(() => {
                button.disabled = false;
                button.textContent = '🔒 Change Password';
            });
        }
        
        // Change self password function (for pricing managers)
        function changeSelfPassword() {
            if (!IS_PRICING_MANAGER) {
                alert('You do not have permission to change passwords.');
                return;
            }
            
            const oldPasswordInput = document.getElementById('old-password-self');
            const newPasswordInput = document.getElementById('new-password-self');
            const messageDiv = document.getElementById('message-self');
            const oldPassword = oldPasswordInput.value.trim();
            const newPassword = newPasswordInput.value.trim();
            
            // Validate inputs
            if (!oldPassword) {
                messageDiv.textContent = '❌ Please enter your current password';
                messageDiv.className = 'password-message error';
                messageDiv.style.display = 'block';
                return;
            }
            
            if (newPassword.length < 6) {
                messageDiv.textContent = '❌ Password must be at least 6 characters long';
                messageDiv.className = 'password-message error';
                messageDiv.style.display = 'block';
                return;
            }
            
            // Disable button during request
            const button = newPasswordInput.nextElementSibling;
            button.disabled = true;
            button.textContent = '⏳ Updating...';
            
            // Send to server
            fetch('change_password.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    username: '<?php echo htmlspecialchars($username); ?>',
                    oldPassword: oldPassword,
                    newPassword: newPassword,
                    isSelfChange: true
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    messageDiv.textContent = '✅ Password updated successfully!';
                    messageDiv.className = 'password-message success';
                    messageDiv.style.display = 'block';
                    oldPasswordInput.value = '';
                    newPasswordInput.value = '';
                    
                    // Hide message after 3 seconds
                    setTimeout(() => {
                        messageDiv.style.display = 'none';
                    }, 3000);
                } else {
                    messageDiv.textContent = '❌ Error: ' + (data.message || 'Failed to update password');
                    messageDiv.className = 'password-message error';
                    messageDiv.style.display = 'block';
                }
            })
            .catch(error => {
                messageDiv.textContent = '❌ Error: ' + error.message;
                messageDiv.className = 'password-message error';
                messageDiv.style.display = 'block';
            })
            .finally(() => {
                button.disabled = false;
                button.textContent = '🔒 Change My Password';
            });
        }
    </script>
</body>
</html>

