<?php
session_start();
require_once 'users.php';

// Check if user is logged in
if (!isLoggedIn()) {
    header('Location: login.php');
    exit;
}

$username = $_SESSION['username'];
$role = $_SESSION['role'];
$isPricingManager = hasRole('pricing_manager');

// Load base prices
require_once '../load_base_prices.php';
$basePriceMap = loadBasePrices();

// Load selling prices
$sellingPricesFile = 'selling_prices.json';
$sellingPriceMap = [];
if (file_exists($sellingPricesFile)) {
    $sellingPricesJson = file_get_contents($sellingPricesFile);
    $sellingPrices = json_decode($sellingPricesJson, true);
    if (is_array($sellingPrices)) {
        foreach ($sellingPrices as $item) {
            if (isset($item['id']) && isset($item['selling_price'])) {
                $sellingPriceMap[$item['id']] = $item['selling_price'];
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Debug Test - Bait Alamer</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            padding: 20px;
            background: #f5f5f5;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            padding: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #2c3e50;
            border-bottom: 3px solid #e67e22;
            padding-bottom: 10px;
        }
        .info-box {
            background: #e3f2fd;
            padding: 15px;
            border-radius: 5px;
            margin: 20px 0;
            border-left: 4px solid #2196f3;
        }
        .success {
            background: #e8f5e9;
            border-left-color: #4caf50;
        }
        .error {
            background: #ffebee;
            border-left-color: #f44336;
        }
        .warning {
            background: #fff3cd;
            border-left-color: #ffc107;
        }
        button {
            background: #e67e22;
            color: white;
            border: none;
            padding: 12px 30px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 16px;
            margin: 10px 5px;
        }
        button:hover {
            background: #d35400;
        }
        .code {
            background: #f4f4f4;
            padding: 10px;
            border-radius: 4px;
            font-family: monospace;
            overflow-x: auto;
        }
        #console {
            background: #1e1e1e;
            color: #00ff00;
            padding: 15px;
            border-radius: 5px;
            font-family: monospace;
            height: 300px;
            overflow-y: auto;
            margin-top: 20px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔧 Debug Test Page</h1>
        
        <div class="info-box">
            <strong>👤 User:</strong> <?php echo htmlspecialchars($username); ?><br>
            <strong>🎭 Role:</strong> <?php echo htmlspecialchars($role); ?><br>
            <strong>🔐 Is Pricing Manager:</strong> <?php echo $isPricingManager ? 'Yes' : 'No'; ?>
        </div>

        <div class="info-box">
            <strong>📊 Base Prices Loaded:</strong> <?php echo count($basePriceMap); ?> items<br>
            <strong>💰 Selling Prices Loaded:</strong> <?php echo count($sellingPriceMap); ?> items
        </div>

        <h2>Test Functions</h2>
        
        <button onclick="testEmailSend()">📧 Test Email Send</button>
        <button onclick="testBasePrices()">💰 Test Base Prices</button>
        <button onclick="testJSVariables()">🔍 Test JS Variables</button>
        <button onclick="clearConsole()">🗑️ Clear Console</button>
        <a href="customer.php" style="display: inline-block; background: #3498db; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; margin: 10px 5px;">← Back to Calculator</a>

        <h2>Console Output</h2>
        <div id="console"></div>
    </div>

    <script>
        // Inject PHP variables
        const USER_ROLE = '<?php echo $role; ?>';
        const IS_PRICING_MANAGER = <?php echo $isPricingManager ? 'true' : 'false'; ?>;
        const BASE_PRICE_MAP = <?php echo json_encode($basePriceMap); ?>;
        const SELLING_PRICE_MAP = <?php echo json_encode($sellingPriceMap); ?>;

        function log(message, type = 'info') {
            const console = document.getElementById('console');
            const timestamp = new Date().toLocaleTimeString();
            const colors = {
                info: '#00ff00',
                success: '#00ff00',
                error: '#ff0000',
                warning: '#ffff00'
            };
            console.innerHTML += `<div style="color: ${colors[type]}">[${timestamp}] ${message}</div>`;
            console.scrollTop = console.scrollHeight;
        }

        function clearConsole() {
            document.getElementById('console').innerHTML = '';
            log('Console cleared');
        }

        function testJSVariables() {
            log('=== Testing JavaScript Variables ===');
            log('USER_ROLE: ' + USER_ROLE);
            log('IS_PRICING_MANAGER: ' + IS_PRICING_MANAGER);
            log('BASE_PRICE_MAP keys: ' + Object.keys(BASE_PRICE_MAP).length);
            log('SELLING_PRICE_MAP keys: ' + Object.keys(SELLING_PRICE_MAP).length);
            log('Sample BASE_PRICE_MAP:', 'info');
            log(JSON.stringify(Object.fromEntries(Object.entries(BASE_PRICE_MAP).slice(0, 3)), null, 2));
        }

        function testBasePrices() {
            log('=== Testing Base Prices ===');
            const sampleIds = [1, 2, 3, 4, 5];
            sampleIds.forEach(id => {
                const basePrice = BASE_PRICE_MAP[id];
                const sellingPrice = SELLING_PRICE_MAP[id];
                log(`Item ${id}: Base=${basePrice}, Selling=${sellingPrice}`);
            });
        }

        async function testEmailSend() {
            log('=== Testing Email Send ===', 'info');
            log('Preparing test order data...');

            const testOrderData = {
                orderNumber: 'TEST-' + Date.now(),
                date: new Date().toLocaleString(),
                customerName: 'Debug Test Customer',
                siteLocation: 'Debug Test Location',
                generalNotes: 'This is a test order from debug page',
                sellingGrandTotal: 500000,
                baseGrandTotal: 400000,
                items: [
                    {
                        category: 'Gypsum Board Items',
                        name: 'Test Item 1',
                        unit: 'm²',
                        inputs: 'L: 10m W: 5m',
                        adjustedQty: '50',
                        sellingPrice: 10000,
                        sellingLineTotal: 500000,
                        basePrice: 8000,
                        baseLineTotal: 400000
                    }
                ]
            };

            log('Order Data: ' + JSON.stringify(testOrderData, null, 2));
            log('Sending POST request to send_order_email.php...', 'warning');

            try {
                const response = await fetch('send_order_email.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(testOrderData)
                });

                log('Response Status: ' + response.status + ' ' + response.statusText);

                if (!response.ok) {
                    const errorText = await response.text();
                    log('HTTP Error: ' + errorText, 'error');
                    throw new Error('HTTP ' + response.status);
                }

                const result = await response.json();
                log('Server Response: ' + JSON.stringify(result, null, 2));

                if (result.success) {
                    log('✅ Email sent successfully!', 'success');
                    log('Order Number: ' + result.orderNumber, 'success');
                    log('Email Sent: ' + result.emailSent, 'success');
                } else {
                    log('❌ Email failed: ' + result.message, 'error');
                }
            } catch (error) {
                log('❌ Error: ' + error.message, 'error');
                log('Stack: ' + error.stack, 'error');
            }
        }

        // Initial log
        log('Debug page loaded successfully', 'success');
        log('User: <?php echo $username; ?> (<?php echo $role; ?>)');
    </script>
</body>
</html>

