<?php
session_start();
require_once 'users.php';

// Set JSON response header
header('Content-Type: application/json');

// Check if user is logged in and has pricing manager role
if (!isLoggedIn()) {
    echo json_encode(['success' => false, 'message' => 'Not authenticated']);
    exit;
}

if (!hasRole('pricing_manager')) {
    echo json_encode(['success' => false, 'message' => 'Insufficient permissions']);
    exit;
}

// Get JSON input
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!isset($data['prices']) || !is_array($data['prices'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid data format']);
    exit;
}

// Load current selling prices
$sellingPricesFile = 'selling_prices.json';
$sellingPrices = [];

if (file_exists($sellingPricesFile)) {
    $sellingPricesJson = file_get_contents($sellingPricesFile);
    $sellingPrices = json_decode($sellingPricesJson, true);
}

// If no file exists, initialize empty array
if (!is_array($sellingPrices)) {
    $sellingPrices = [];
}

// Update prices - ensure we have entries for all item IDs
$updatedCount = 0;
foreach ($data['prices'] as $itemId => $newPrice) {
    $newPrice = floatval($newPrice);
    if ($newPrice >= 0) {
        $itemId = intval($itemId);
        
        // Find existing entry or create new one
        $found = false;
        foreach ($sellingPrices as &$item) {
            if ($item['id'] === $itemId) {
                $item['selling_price'] = $newPrice;
                $found = true;
                $updatedCount++;
                break;
            }
        }
        
        // If not found, add new entry
        if (!$found) {
            $sellingPrices[] = [
                'id' => $itemId,
                'selling_price' => $newPrice
            ];
            $updatedCount++;
        }
    }
}

// Sort by ID for consistency
usort($sellingPrices, function($a, $b) {
    return $a['id'] - $b['id'];
});

// Save back to file
$newJson = json_encode($sellingPrices, JSON_PRETTY_PRINT);
if (file_put_contents($sellingPricesFile, $newJson) !== false) {
    // Log the update
    $logEntry = date('Y-m-d H:i:s') . " - User: {$_SESSION['username']} updated {$updatedCount} price(s)\n";
    file_put_contents('price_updates.log', $logEntry, FILE_APPEND);
    
    echo json_encode([
        'success' => true, 
        'message' => "Successfully updated {$updatedCount} price(s)",
        'updated_count' => $updatedCount
    ]);
} else {
    echo json_encode(['success' => false, 'message' => 'Failed to save prices file']);
}
?>

