<?php
// Prevent any output before JSON
error_reporting(E_ALL);
ini_set('display_errors', 0); // Don't display errors in output
ini_set('log_errors', 1); // Log errors instead

session_start();
require_once 'users.php';

// Load PHPMailer
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

require 'PHPMailer/src/Exception.php';
require 'PHPMailer/src/PHPMailer.php';
require 'PHPMailer/src/SMTP.php';

// Load email configuration
$emailConfig = require 'email_config.php';

// Check if user is logged in
if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

// Load base prices
require_once '../load_base_prices.php';
$basePriceMap = loadBasePrices();

// Get POST data
$data = json_decode(file_get_contents('php://input'), true);

if (!$data) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid data']);
    exit;
}

// Extract order information
$orderData = [
    'orderNumber' => $data['orderNumber'] ?? 'N/A',
    'date' => $data['date'] ?? date('Y-m-d H:i:s'),
    'customerName' => $data['customerName'] ?? 'N/A',
    'siteLocation' => $data['siteLocation'] ?? 'N/A',
    'preparedBy' => $_SESSION['username'] ?? 'Unknown',
    'role' => $_SESSION['role'] ?? 'Unknown',
    'items' => $data['items'] ?? [],
    'sellingGrandTotal' => $data['sellingGrandTotal'] ?? 0,
    'baseGrandTotal' => $data['baseGrandTotal'] ?? 0,
    'generalNotes' => $data['generalNotes'] ?? ''
];

// Calculate profit
$totalProfit = $orderData['sellingGrandTotal'] - $orderData['baseGrandTotal'];
$profitMargin = $orderData['sellingGrandTotal'] > 0 
    ? ($totalProfit / $orderData['sellingGrandTotal']) * 100 
    : 0;

// Generate Internal Cost PDF HTML
$internalPdfHtml = generateInternalPDF($orderData, $basePriceMap, $totalProfit, $profitMargin);
$pdfFilename = 'internal_cost_' . $orderData['orderNumber'] . '.html';

// Prepare email using PHPMailer
$mail = new PHPMailer(true);

try {
    // Server settings
    $mail->isSMTP();
    $mail->Host       = $emailConfig['smtp_host'];
    $mail->SMTPAuth   = $emailConfig['smtp_auth'];
    $mail->Username   = $emailConfig['smtp_username'];
    $mail->Password   = $emailConfig['smtp_password'];
    $mail->SMTPSecure = ($emailConfig['smtp_encryption'] === 'ssl') ? PHPMailer::ENCRYPTION_SMTPS : PHPMailer::ENCRYPTION_STARTTLS;
    $mail->Port       = $emailConfig['smtp_port'];
    $mail->CharSet    = 'UTF-8';
    
    // Recipients
    $mail->setFrom($emailConfig['from_email'], $emailConfig['from_name']);
    $mail->addAddress($emailConfig['to_email'], $emailConfig['to_name']);
    $mail->addReplyTo($emailConfig['from_email'], $emailConfig['from_name']);
    
    // Subject
    $mail->Subject = 'Bait Alamer Order #' . $orderData['orderNumber'] . ' – Internal Summary';
    
    // Email body (plain text)
    $emailBody = "Order Details:\n";
    $emailBody .= "━━━━━━━━━━━━━━━━━━━━━━━━\n";
    $emailBody .= "Order #: " . $orderData['orderNumber'] . "\n";
    $emailBody .= "Date: " . $orderData['date'] . "\n";
    $emailBody .= "Customer: " . $orderData['customerName'] . "\n";
    $emailBody .= "Site: " . $orderData['siteLocation'] . "\n";
    $emailBody .= "Prepared By: " . $orderData['preparedBy'] . " (" . $orderData['role'] . ")\n\n";
    
    $emailBody .= "Financial Summary:\n";
    $emailBody .= "━━━━━━━━━━━━━━━━━━━━━━━━\n";
    $emailBody .= "Customer Grand Total: " . number_format($orderData['sellingGrandTotal'], 0) . " IQD\n";
    $emailBody .= "Base Grand Total: " . number_format($orderData['baseGrandTotal'], 0) . " IQD\n";
    $emailBody .= "Total Profit: " . number_format($totalProfit, 0) . " IQD\n";
    $emailBody .= "Profit Margin: " . number_format($profitMargin, 2) . "%\n\n";
    
    if (!empty($orderData['generalNotes'])) {
        $emailBody .= "General Notes:\n" . $orderData['generalNotes'] . "\n\n";
    }
    
    $emailBody .= "See attached HTML file for detailed breakdown.\n";
    $emailBody .= "You can open it in any browser and print/save as PDF.\n\n";
    $emailBody .= "---\n";
    $emailBody .= "Bait Alamer Pricing Calculator\n";
    $emailBody .= "https://baitalamer.alnubala.com/Bait-alamer/\n";
    
    $mail->Body = $emailBody;
    
    // Attach HTML file
    $mail->addStringAttachment($internalPdfHtml, $pdfFilename, 'base64', 'text/html');
    
    // Send email
    $emailSent = $mail->send();
    
} catch (Exception $e) {
    $emailSent = false;
    $errorDetails = [
        'error_info' => $mail->ErrorInfo,
        'exception_message' => $e->getMessage(),
        'exception_trace' => $e->getTraceAsString(),
        'timestamp' => date('Y-m-d H:i:s')
    ];
    error_log("Email Error: " . json_encode($errorDetails));
    // Also write to debug file
    file_put_contents('email_debug.log', 
        date('Y-m-d H:i:s') . " | ERROR: " . $mail->ErrorInfo . " | Exception: " . $e->getMessage() . "\n", 
        FILE_APPEND
    );
}

// No temp file cleanup needed with PHPMailer (uses string attachment)

// Log the order
$logEntry = date('Y-m-d H:i:s') . " | Order #{$orderData['orderNumber']} | {$orderData['preparedBy']} ({$orderData['role']}) | Customer: {$orderData['customerName']} | Total: {$orderData['sellingGrandTotal']} IQD | Profit: {$totalProfit} IQD | Email: " . ($emailSent ? 'Sent' : 'Failed') . "\n";
file_put_contents('orders.log', $logEntry, FILE_APPEND);

// Return response
echo json_encode([
    'success' => $emailSent,
    'message' => $emailSent ? 'Order processed and email sent successfully' : 'Order processed but email failed',
    'orderNumber' => $orderData['orderNumber'],
    'emailSent' => $emailSent
]);

function generateInternalPDF($orderData, $basePriceMap, $totalProfit, $profitMargin) {
    $html = '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Internal Cost Summary - Order #' . htmlspecialchars($orderData['orderNumber']) . '</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: "Segoe UI", Arial, sans-serif;
            padding: 20px;
            background: #f5f5f5;
            color: #333;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            padding: 40px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        
        .internal-warning {
            background: #ff4444;
            color: white;
            padding: 15px;
            text-align: center;
            font-weight: bold;
            font-size: 18px;
            margin-bottom: 30px;
            border-radius: 5px;
        }
        
        .header {
            text-align: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 3px solid #e67e22;
        }
        
        .header h1 {
            color: #2c3e50;
            font-size: 28px;
            margin-bottom: 10px;
        }
        
        .header .order-number {
            font-size: 20px;
            color: #e67e22;
            font-weight: bold;
        }
        
        .order-info {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
            margin-bottom: 30px;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 5px;
        }
        
        .order-info .info-item {
            padding: 8px 0;
        }
        
        .order-info .info-label {
            font-weight: bold;
            color: #555;
            display: inline-block;
            min-width: 120px;
        }
        
        .order-info .info-value {
            color: #222;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
        }
        
        thead {
            background: #34495e;
            color: white;
        }
        
        th, td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }
        
        th {
            font-weight: 600;
            text-transform: uppercase;
            font-size: 11px;
        }
        
        tbody tr:hover {
            background: #f8f9fa;
        }
        
        .text-right {
            text-align: right !important;
        }
        
        .text-center {
            text-align: center !important;
        }
        
        .price-column {
            font-family: "Courier New", monospace;
            font-weight: bold;
        }
        
        .selling-price {
            color: #27ae60;
        }
        
        .base-price {
            color: #e67e22;
        }
        
        .profit-column {
            color: #2980b9;
            font-weight: bold;
        }
        
        .summary-box {
            margin-top: 30px;
            padding: 25px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 8px;
        }
        
        .summary-box h2 {
            margin-bottom: 20px;
            font-size: 22px;
        }
        
        .summary-row {
            display: flex;
            justify-content: space-between;
            padding: 10px 0;
            font-size: 16px;
            border-bottom: 1px solid rgba(255,255,255,0.2);
        }
        
        .summary-row:last-child {
            border-bottom: none;
            font-size: 18px;
            font-weight: bold;
            margin-top: 10px;
            padding-top: 15px;
            border-top: 2px solid white;
        }
        
        .summary-label {
            font-weight: 600;
        }
        
        .summary-value {
            font-family: "Courier New", monospace;
        }
        
        .category-header {
            background: #ecf0f1;
            font-weight: bold;
            color: #2c3e50;
            padding: 15px 12px !important;
            font-size: 14px;
        }
        
        .notes-section {
            margin-top: 30px;
            padding: 20px;
            background: #fff3cd;
            border-left: 4px solid #ffc107;
            border-radius: 5px;
        }
        
        .notes-section h3 {
            margin-bottom: 10px;
            color: #856404;
        }
        
        .notes-section p {
            color: #856404;
            line-height: 1.6;
        }
        
        .footer {
            margin-top: 40px;
            padding-top: 20px;
            border-top: 2px solid #ecf0f1;
            text-align: center;
            color: #7f8c8d;
            font-size: 12px;
        }
        
        @media print {
            body {
                background: white;
            }
            .container {
                box-shadow: none;
                padding: 0;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="internal-warning">
            ⚠️ INTERNAL COPY – NOT FOR CUSTOMER ⚠️
        </div>
        
        <div class="header">
            <h1>Bait Alamer - Internal Cost Summary</h1>
            <div class="order-number">Order #' . htmlspecialchars($orderData['orderNumber']) . '</div>
        </div>
        
        <div class="order-info">
            <div class="info-item">
                <span class="info-label">Customer:</span>
                <span class="info-value">' . htmlspecialchars($orderData['customerName']) . '</span>
            </div>
            <div class="info-item">
                <span class="info-label">Date:</span>
                <span class="info-value">' . htmlspecialchars($orderData['date']) . '</span>
            </div>
            <div class="info-item">
                <span class="info-label">Site Location:</span>
                <span class="info-value">' . htmlspecialchars($orderData['siteLocation']) . '</span>
            </div>
            <div class="info-item">
                <span class="info-label">Prepared By:</span>
                <span class="info-value">' . htmlspecialchars($orderData['preparedBy']) . ' (' . htmlspecialchars($orderData['role']) . ')</span>
            </div>
        </div>
        
        <table>
            <thead>
                <tr>
                    <th>Item</th>
                    <th class="text-center">Unit</th>
                    <th class="text-center">Adj. Qty</th>
                    <th class="text-right">Selling Price</th>
                    <th class="text-right">Selling Total</th>
                    <th class="text-right">Base Price</th>
                    <th class="text-right">Base Total</th>
                    <th class="text-right">Item Profit</th>
                </tr>
            </thead>
            <tbody>';
    
    $currentCategory = '';
    foreach ($orderData['items'] as $item) {
        // Add category header row
        if ($item['category'] !== $currentCategory) {
            $currentCategory = $item['category'];
            $html .= '<tr><td colspan="8" class="category-header">' . htmlspecialchars($currentCategory) . '</td></tr>';
        }
        
        $itemProfit = $item['sellingLineTotal'] - $item['baseLineTotal'];
        
        $html .= '<tr>
            <td>' . htmlspecialchars($item['name']) . '</td>
            <td class="text-center">' . htmlspecialchars($item['unit']) . '</td>
            <td class="text-center">' . htmlspecialchars($item['adjustedQty']) . ' ' . htmlspecialchars($item['unit']) . '</td>
            <td class="text-right price-column selling-price">' . number_format($item['sellingPrice'], 0) . '</td>
            <td class="text-right price-column selling-price">' . number_format($item['sellingLineTotal'], 0) . '</td>
            <td class="text-right price-column base-price">' . number_format($item['basePrice'], 0) . '</td>
            <td class="text-right price-column base-price">' . number_format($item['baseLineTotal'], 0) . '</td>
            <td class="text-right profit-column">' . number_format($itemProfit, 0) . '</td>
        </tr>';
    }
    
    $html .= '</tbody>
        </table>
        
        <div class="summary-box">
            <h2>📊 Financial Summary</h2>
            <div class="summary-row">
                <span class="summary-label">Customer Grand Total:</span>
                <span class="summary-value">' . number_format($orderData['sellingGrandTotal'], 0) . ' IQD</span>
            </div>
            <div class="summary-row">
                <span class="summary-label">Base Grand Total:</span>
                <span class="summary-value">' . number_format($orderData['baseGrandTotal'], 0) . ' IQD</span>
            </div>
            <div class="summary-row">
                <span class="summary-label">Total Profit:</span>
                <span class="summary-value">' . number_format($totalProfit, 0) . ' IQD</span>
            </div>
            <div class="summary-row">
                <span class="summary-label">Profit Margin:</span>
                <span class="summary-value">' . number_format($profitMargin, 2) . '%</span>
            </div>
        </div>';
    
    if (!empty($orderData['generalNotes'])) {
        $html .= '<div class="notes-section">
            <h3>📝 General Notes</h3>
            <p>' . nl2br(htmlspecialchars($orderData['generalNotes'])) . '</p>
        </div>';
    }
    
    $html .= '
        <div class="footer">
            <p>Generated by Bait Alamer Pricing Calculator</p>
            <p>https://baitalamer.alnubala.com/Bait-alamer/</p>
            <p>This document contains confidential pricing information</p>
        </div>
    </div>
</body>
</html>';
    
    return $html;
}
?>

